#ifndef __LIBIDT_NEO2_H___
# define __LIBIDT_NEO2_H___


# define IN
# define OUT
# define IN_OUT
# include "IDTDef.h"

/**
 * Define the USB hot-plug callback function to monitor the info when
 * plug in/out the reader. <br/> It should be registered using the
 * registerHotplugCallBk, The first integer parameter is device type,
 * and the second integer parameter is either 0: Device Plugged Out or
 * 1: Device Plugged In
 */
typedef void (* pMessageHotplug)(int,
  int);

/**
 * Define the send command callback function to monitor the sending
 * command into the reader. <br/> It should be registered using the
 * registerLogCallBk
 */
typedef void (* pSendDataLog)(BYTE *,
  int);

/**
 * Define the read response callback function to monitor the reading
 * response from the reader. <br/> It should be registered using the
 * registerLogCallBk
 */
typedef void (* pReadDataLog)(BYTE *,
  int);

/**
 * Define the EMV callback function to get the transaction
 * message/data/result. <br/> It should be registered using the
 * emv_registerCallBk
 */
typedef void (* pEMV_callBack)(int,
  int,
  BYTE *,
  int,
  IDTTransactionData *,
  EMV_Callback *,
  int);

/**
 * Define the Worldpay callback function to get the transaction
 * message/data/result. <br/>
 */
typedef void (* pWP_callBack)(char *,
  int,
  int);

/**
 * Define the firmware update callback function to get the firmware
 * update status <br/> It should be registered using the
 * device_registerFWCallBk
 */
typedef void (* pFW_callBack)(int,
  int,
  int,
  int,
  int);

/**
 * Define the MSR callback function to get the MSR card data <br/>
 * It should be registered using the msr_registerCallBk, this callback function is for backward compatibility
 */
typedef void (* pMSR_callBack)(int, IDTMSRData);

/**
 * Define the MSR callback function to get pointer to the MSR card data <br/>
 * It should be registered using the msr_registerCallBk, this callback function is recommended instead of pMSR_callBack
 */
typedef void (* pMSR_callBackp)(int, IDTMSRData *);

/**
 * Define the PINPad callback function to get the input PIN Pad data <br/>
 * It should be registered using the pin_registerCallBk,
 */
typedef void (* pPIN_callBack)(int,
  IDTPINData *);

/**
 * Define the camera callback function to get the image data <br/>
 * It should be registered using the device_registerCameraCallBk,
 */
typedef void (* pCMR_callBack)(int,
  IDTCMRData *);

/**
 * Define the card status and front switch callback function to get card and front switch status <br/>
 * It should be registered using the device_registerCardStatusFrontSwitchCallBk,
 */
typedef void (* pCSFS_callBack)(BYTE status);

/**
 * Define the LCD callback function to get the input LCDItem <br/>
 * It should be registered using the lcd_registerCallBk,
 */
typedef void (* pLCD_callBack)(int,
  IDTLCDItem *);

/**
 * Define the RKI callback function to get the status of the RKI <br/>
 * It should be registered using the device_registerRKICallBk,
 */
typedef void (* pRKI_callBack)(int,
  char *);

/**
 * Define the comm callback function to get FTP file transfer status <br/>
 * It should be passed as a parameter in a FTP request,
 * Signature (int, int, int) = response code, current block, total blocks
 * RESPONSE CODES:
 *              100 = FILE DOWNLOAD STARTED
 *              101 = FILE BLOCK XX OF XX RECEIVED
 *              102 = FILE DOWNLOAD COMPLETED
 *              103 = FILE DOWNLOAD TERMINATED PREMATURELY
 *
 */
typedef void (* ftpComm_callBack)(int,
  int,
  int);

/**
 * Define the comm callback function to get the async url data <br/>
 * It should be registered using the comm_registerHTTPCallback
 */
typedef void (* httpComm_callBack)(BYTE *,
  int);

/**
 * Define the comm callback function to receive the V4 Protocol packets
 * received by the device from an external source (IP/USB/RS-232)
 * It should be registered using the comm_registerV4Callback,
 * Data callback will contain command, sub-command, and data from V4 packet
 */
typedef void (* v4Comm_callBack)(BYTE,
  BYTE,
  BYTE *,
  int);


# ifdef __cplusplus
extern "C" {
# endif

/**
 * To register the USB HID hot-plug callback function which implemented
 * in the application to monitor the hotplug message from the SDK.
 */
void
registerHotplugCallBk(pMessageHotplug pMsgHotplug);

/**
 * To register the log callback function which implemented in the
 * application to monitor sending/reading data between application and
 * reader.
 */
void
registerLogCallBk(pSendDataLog pFSend,
  pReadDataLog                 pFRead);

/**
 * To register the firmware update callback function to get the
 * firmware update processing response.  (Pass NULL to disable the callback.)
 */
void
device_registerFWCallBk(pFW_callBack pFWf);

/**
 * To register the camera callback function to get the image data.  (Pass NULL to disable the callback.)
 */
void
device_registerCameraCallBk(pCMR_callBack pCMRf);

/**
 * To register the card status and front switch callback function to get status.  (Pass NULL to disable the callback.)
 */
void
device_registerCardStatusFrontSwitchCallBk(pCSFS_callBack pCSFSf);

/**
 * To register the RKI callback function to get the
 * RKI status.  (Pass NULL to disable the callback.)
 */
void
device_registerRKICallBk(pRKI_callBack pRKIf);

/**
 * To register the emv callback function to get the EMV processing response.  (Pass NULL to disable the callback.)
 */
void
emv_registerCallBk(pEMV_callBack pEMVf);

/**
 * To register the loyalty callback function to get the EMV processing response.  (Pass NULL to disable the callback.)
 *     Only for VP6800
 */
void
loyalty_registerCallBk(pEMV_callBack pEMVf);

/**
 * To register the msr callback function to get the MSR card data.  (Pass NULL to disable the callback.)
 */
void
msr_registerCallBk(pMSR_callBack pMSRf);

/**
 * To register the msr callback function to get the MSR card data pointer.  (Pass NULL to disable the callback.)
 */
void
msr_registerCallBkp(pMSR_callBackp pMSRf);

///**
// *To register the loyalty MSR callback function to get the MSR card data.  (Pass NULL to disable the callback.)
// */
// void loyalty_registerMSRCallBk(pMSR_callBack pMSRf);
//
///**
// *To register the loyalty MSR callback function to get the MSR card data pointer.  (Pass NULL to disable the callback.)
// */
// void loyalty_registerMSRCallBkp(pMSR_callBackp pMSRf);

/**
 * To register the ctls callback function to get the MSR card data.  (Pass NULL to disable the callback.)
 */
void
ctls_registerCallBk(pMSR_callBack pCTLSf);

/**
 * To register the ctls callback function to get the MSR card data pointer.  (Pass NULL to disable the callback.)
 */
void
ctls_registerCallBkp(pMSR_callBackp pCTLSf);

/**
 * To register the pin callback function to get the PINPad data.  (Pass NULL to disable the callback.)
 */
void
pin_registerCallBk(pPIN_callBack pPINf);

/**
 * To register the lcd callback function to get the LCDItem.  (Pass NULL to disable the callback.)
 */
void
lcd_registerCallBk(pLCD_callBack pLCDf);

/**
 * Register Comm HTTP Async Callback
 *
 * @param cBack - HTTP Comm callback
 */
void
comm_registerHTTPCallback(httpComm_callBack cBack);

/**
 * Register External V4 Protocol commands Callback
 *
 * @param cBack - V4 Protocol Comm callback
 */
void
comm_registerV4Callback(v4Comm_callBack cBack);

/**
 * To Get SDK version
 *@return return the SDK version string
 */
char *
SDK_Version();

/**
 * Set the path to use when searching for ID TECH's libraries.
 * If this is not set, the libraries will be searched for with the system's default procedures.
 *
 * @param absoluteLibraryPath The absolute path to ID TECH's libraries.
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
setAbsoluteLibraryPath(const char * absoluteLibraryPath);

/**
 * Set the path to the config xml file(s) if any <br/>
 *
 * @param path The path to the config xml files (such as "NEO2_Devices.xml"
 * which contains the information of NEO2 devices).
 *    Only need to specify the path to the folder which contains the config files.
 *    File names are not needed.
 *    The maximum length of path is 200 characters including the '\0' at the end.
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_setConfigPath(const char * path);

/**
 * Pass the content of the config xml file ("NEO2_Devices.xml") as a
 * string to the SDK instead of reading the config xml file by the SDK
 * It needs to be called before device_init(), otherwise the SDK will
 * try to read the config xml file.
 *
 * @param configs The content read from the config xml file ("NEO2_Devices.xml"
 *  which contains the information of NEO2 devices).
 * @param len The length of the string configs.  The maximum length is
 *  5000 bytes.
 */
int
device_setNEO2DevicesConfigs(IN const char * configs,
  IN int                                     len);

/**
 * Initial the device by USB<br/>
 * It will detect the device and trying connect. <br/>
 * The connect status can be checked by device_isConnected(). <br/>
 * Note: after the function returns success, the function device_setCurrentDevice() has to be called to set the device type.
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_init();

/**
 * Set the generation for the NEO 2 or 3 readers<br/>
 *
 * @param gen The generation of the NEO 2/3 readers.  2: NEO 2, 3: NEO 3
 *
 */
void
device_setNEOGen(int gen);

/**
 * Set the alternative device type for the NEO 2 or 3 readers<br/>
 *
 * @param alt The alternative device type of the NEO 2/3 readers.
 *
 */
void
device_setNEOAltDevice(int alt);

/**
 * Initial the device by RS232<br/>
 * It will try to connect to the device with provided deviceType, port_number, and brate. <br/>
 * @param deviceType Device to connect to
 * @param port_number Port number of the device
 *
 *   Port nr. |  Linux  | Windows
 *
 *  -----------------------------
 *
 *  | 0       | ttyS0   | COM1  |
 *
 *  | 1       | ttyS1   | COM2  |
 *
 *  | 2       | ttyS2   | COM3  |
 *
 *  | 3       | ttyS3   | COM4  |
 *
 *  | 4       | ttyS4   | COM5  |
 *
 *  | 5       | ttyS5   | COM6  |
 *
 *  | 6       | ttyS6   | COM7  |
 *
 *  | 7       | ttyS7   | COM8  |
 *
 *  | 8       | ttyS8   | COM9  |
 *
 *  | 9       | ttyS9   | COM10 |
 *
 *  | 10      | ttyS10  | COM11 |
 *
 *  | 11      | ttyS11  | COM12 |
 *
 *  | 12      | ttyS12  | COM13 |
 *
 *  | 13      | ttyS13  | COM14 |
 *
 *  | 14      | ttyS14  | COM15 |
 *
 *  | 15      | ttyS15  | COM16 |
 *
 *  | 16      | ttyUSB0 | n.a.  |
 *
 *  | 17      | ttyUSB1 | n.a.  |
 *
 *  | 18      | ttyUSB2 | n.a.  |
 *
 *  | 19      | ttyUSB3 | n.a.  |
 *
 *  | 20      | ttyUSB4 | n.a.  |
 *
 *  | 21      | ttyUSB5 | n.a.  |
 *
 *  | 22      | ttyAMA0 | n.a.  |
 *
 *  | 23      | ttyAMA1 | n.a.  |
 *
 *  | 24      | ttyACM0 | n.a.  |
 *
 *  | 25      | ttyACM1 | n.a.  |
 *
 *  | 26      | rfcomm0 | n.a.  |
 *
 *  | 27      | rfcomm1 | n.a.  |
 *
 *  | 28      | ircomm0 | n.a.  |
 *
 *  | 29      | ircomm1 | n.a.  |
 *
 *  | 30      | cuau0   | n.a.	|
 *
 *  | 31      | cuau1   | n.a.  |
 *
 *  | 32      | cuau2   | n.a.  |
 *
 *  | 33      | cuau3   | n.a.  |
 *
 *  | 34      | cuaU0   | n.a.  |
 *
 *  | 35      | cuaU1   | n.a.  |
 *
 *  | 36      | cuaU2   | n.a.  |
 *
 *  | 37      | cuaU3   | n.a.  |
 *
 *  | 38      | ttyIDG  | n.a.  |
 * @param brate Bitrate of the device
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
rs232_device_init(int deviceType, int port_number, int brate);

/**
 * Set the timeout value for opening the com port <br/>
 *
 * @param timeout should be set to greater than 0 in seconds, otherwise there will be no timeout for opening the com port
 *
 */
void
set_open_com_port_timeout(int timeout);

/**
 * Sets the current device to talk to <br/>
 * The connect status can be checked by device_isConnected(). <br/>
 * Note: the file "NEO2_Devices.xml" is required for NEO 2 readers
 * @param deviceType Device to connect to
 * @code
 * enum DEVICE_TYPE {
 * IDT_DEVICE_UNKNOWN=0,
 * IDT_DEVICE_AUGUSTA_HID,
 * IDT_DEVICE_AUGUSTA_KB,
 * IDT_DEVICE_AUGUSTA_S_HID,
 * IDT_DEVICE_AUGUSTA_S_KB,
 * IDT_DEVICE_AUGUSTA_S_TTK_HID,
 * IDT_DEVICE_SPECTRUM_PRO,
 * IDT_DEVICE_MINISMART_II,
 * IDT_DEVICE_L80,
 * IDT_DEVICE_L100,
 * IDT_DEVICE_UNIPAY,
 * IDT_DEVICE_UNIPAY_I_V,
 * IDT_DEVICE_VP3300_AJ,
 * IDT_DEVICE_KIOSK_III,
 * IDT_DEVICE_KIOSK_III_S,
 * IDT_DEVICE_PIP_READER,
 * IDT_DEVICE_VENDI,
 * IDT_DEVICE_VP3300_USB,
 * IDT_DEVICE_UNIPAY_I_V_TTK,
 * IDT_DEVICE_VP3300_BT,
 * IDT_DEVICE_VP8800,
 * IDT_DEVICE_SREDKEY2_HID,
 * IDT_DEVICE_SREDKEY2_KB,
 * IDT_DEVICE_NEO2,
 * IDT_DEVICE_MINISMART_II_COM = IDT_DEVICE_NEO2+5,
 * IDT_DEVICE_SPECTRUM_PRO_COM,
 * IDT_DEVICE_KIOSK_III_COM,
 * IDT_DEVICE_KIOSK_III_S_COM,
 * IDT_DEVICE_VP3300_COM,
 * IDT_DEVICE_NEO2_COM,
 * IDT_DEVICE_MAX_DEVICES = IDT_DEVICE_NEO2_COM+5
 * };
 *      @endcode
 * @return RETURN_CODE:    1: success, 0: failed
 */
int
device_setCurrentDevice(int deviceType);

/**
 * Check if the device is attached to the USB port
 * The function device_init() must be called before this function.
 * @param deviceType, the device type of the USB device
 * @return 1 if the device is attached, or 0 if the device is not attached
 */
int
device_isAttached(int deviceType);

/**
 * Close the device <br/>
 *
 * @return RETURN_CODE:  0: success, 0x0A: failed
 */
int
device_close();

/**
 * Review the return code description.<br/>
 * @param returnCode  the response result.
 * @param description
 * @retval the string for description of response result
 *
 * - 0: "no error, beginning task";
 * - 1: "no response from reader";
 * - 2: "invalid response data";
 *
 * - 01: " Incorrect Header Tag";
 * - 02: " Unknown Command";
 * - 03: " Unknown Sub-Command";
 * - 04: " CRC Error in Frame";
 * - 05: " Incorrect Parameter";
 * - 06: " Parameter Not Supported";
 * - 07: " Mal-formatted Data";
 * - 08: " Timeout";
 * - 0A: " Failed / NACK";
 * - 0B: " Command not Allowed";
 * - 0C: " Sub-Command not Allowed";
 * - 0D: " Buffer Overflow (Data Length too large for reader buffer)";
 * - 0E: " User Interface Event";
 * - 10: " Need clear firmware(apply in boot loader only)";
 * - 11: " Communication type not supported, VT-1, burst, etc. Need encrypted firmware (apply in boot loader only)";
 * - 12: " Secure interface is not functional or is in an intermediate state.";
 * - 13: " Data field is not mod 8";
 * - 14: " Pad 0x80 not found where expected";
 * - 15: " Specified key type is invalid";
 * - 16: " Could not retrieve key from the SAM (InitSecureComm)";
 * - 17: " Hash code problem";
 * - 18: " Could not store the key into the SAM (InstallKey)";
 * - 19: " Frame is too large";
 * - 1A: " Unit powered up in authentication state but POS must resend the InitSecureComm command";
 * - 1B: " The EEPROM may not be initialized because SecCommInterface does not make sense";
 * - 1C: " Problem encoding APDU Module-Specific Status Codes ";
 * - 20: " Unsupported Index (ILM) SAM Transceiver error - problem communicating with the SAM (Key Mgr)";
 * - 21: " Unexpected Sequence Counter in multiple frames for single bitmap (ILM)Length error in data returned from the SAM (Key Mgr)
 * - 22: " Improper bit map (ILM)";
 * - 23: " Request Online Authorization";
 * - 24: " ViVOCard3 raw data read successful";
 * - 25: " Message index not available (ILM) ViVOcomm activate transaction card type (ViVOcomm)";
 * - 26: " Version Information Mismatch (ILM)";
 * - 27: " Not sending commands in correct index message index (ILM)";
 * - 28: " Time out or next expected message not received (ILM)";
 * - 29: " ILM languages not available for viewing (ILM)";
 * - 2A: " Other language not supported (ILM)";
 * - 41: " from 41 to 4F, Module-specific errors for Key Manager";
 *
 * - 50: " Auto-Switch OK";
 * - 51: " Auto-Switch failed";
 * - 70: " Antenna Error 80h Use another card";
 * - 81: " Insert or swipe card";
 * - 90: " Data encryption Key does not exist";
 * - 91: " Data encryption Key KSN exhausted";
 *
 */
void
device_getIDGStatusCodeString(IN int returnCode,
  OUT char *                         despcrition);

/**
 * Check the device conntected status
 * @return DEVICE_DISCONNECT=0, or DEVICE_CONNECTED = 1
 */
int
device_isConnected();

/**
 * DEPRECATED : please use device_getFirmwareVersion_Len(OUT char* firmwareVersion, IN_OUT int *firmwareVersionLen)
 *
 * Polls device for Firmware Version
 *
 * @param firmwareVersion Response returned of Firmware Version; needs to have at least 128 bytes of memory
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
device_getFirmwareVersion(OUT char * firmwareVersion);

/**
 * Polls device for Firmware Version
 *
 * @param firmwareVersion Response returned of Firmware Version
 * @param firmwareVersionLen Length of Firmware Version
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
device_getFirmwareVersion_Len(OUT char * firmwareVersion, IN_OUT int * firmwareVersionLen);

/**
 * Ping Device
 *
 * Pings the reader.  If connected, returns success.  Otherwise, returns timeout.
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_pingDevice();

/**
 * Control User Interface
 *
 *
 * Controls the User Interface:  Display, Beep, LED
 *
 * @param values Four bytes to control the user interface
 * Byte[0] = LCD Message
 * Messages 00-07 are normally controlled by the reader.
 * - 00h: Idle Message (Welcome)
 * - 01h: Present card (Please Present Card)
 * - 02h: Time Out or Transaction cancel (No Card)
 * - 03h: Transaction between reader and card is in the middle (Processing...)
 * - 04h: Transaction Pass (Thank You)
 * - 05h: Transaction Fail (Fail)
 * - 06h: Amount (Amount $ 0.00 Tap Card)
 * - 07h: Balance or Offline Available funds (Balance $ 0.00) Messages 08-0B are controlled by the terminal
 * - 08h: Insert or Swipe card (Use Chip & PIN)
 * - 09h: Try Again(Tap Again)
 * - 0Ah: Tells the customer to present only one card (Present 1 card only)
 * - 0Bh: Tells the customer to wait for authentication/authorization (Wait)
 * - FFh: indicates the command is setting the LED/Buzzer only.
 * Byte[1] = Beep Indicator
 * - 00h: No beep
 * - 01h: Single beep
 * - 02h: Double beep
 * - 03h: Three short beeps
 * - 04h: Four short beeps
 * - 05h: One long beep of 200 ms
 * - 06h: One long beep of 400 ms
 * - 07h: One long beep of 600 ms
 * - 08h: One long beep of 800 ms
 * Byte[2] = LED Number
 * - 00h: LED 0 (Power LED) 01h: LED 1
 * - 02h: LED 2
 * - 03h: LED 3
 * - FFh: All LEDs
 * Byte[3] = LED Status
 * - 00h: LED Off
 * - 01h: LED On
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_controlUserInterface(IN BYTE * values);

/**
 * Get current active device type
 * @return :  return the device type defined as DEVICE_TYPE in the IDTDef.h
 */
int
device_getCurrentDeviceType();

/**
 * Send a Command to NEO device
 *
 * Sends a command  to the NEO device .
 *
 * @param cmd  command to execute.
 * @param subCmd, sub command to execute.
 *
 * @param data buffer of NEO command data.
 * @param dataLen, the length of the buffer data.
 *
 * @param response Response data
 * @param respLen, the length of Response data
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_SendDataCommandNEO(IN int cmd,
  IN int                         subCmd,
  IN BYTE *                      data,
  IN int                         dataLen,
  OUT BYTE *                     response,
  IN_OUT int *                   respLen);

/**
 * Reboot Device
 * Executes a command to restart the device.
 * - Card data is cleared, resetting card status bits.
 * - Response data of the previous command is cleared.
 * - Resetting firmware.
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_rebootDevice();

/**
 * Stop Audio
 *  This command stops the audio the reader is playing.
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_stopAudio();

/**
 * Play Audio
 *  This command command plays an audio file loaded from the inserted SD card. The VP6800 supports 16bit PCM format .WAV files.
 * @param filename Complete path and file name of the wav file you want to play.
 *   You do not need to specify the root directory. Indicate subdirectories with a forward slash (/).
 * @param filenameLen File Name Length.  Maximum file name length is 30.
 * @param onSD 0: use internal memory (the Maximum audio file in Flash is 5M and only 2 audio file is supported), 1: use SD card
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_playAudio(IN char * fileName, IN int fileNameLen, IN int onSD);

/**
 * Get Audio Volume
 *  This command retrieves the reader’s audio volume.
 * @param Value 0-20, where 0 is silent and 20 is full volume
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_getAudioVolume(OUT BYTE * volume);

/**
 * Set Audio Volume
 *  This command sets the reader’s audio volume.
 * @param Value 0-20, where 0 is silent and 20 is full volume
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_setAudioVolume(IN BYTE volume);

/**
 * Get Camera Parameters
 *  This command is used to get the camera parameters (e.g., auto/fixed focal length as focus).
 * @param isAutoFocus: 0: fixed focus, 1: auto focus
 * @param focalLength: focal length Value 0x00-0xFF, where 0x00 is the farthest, 0xFF is nearest.  Not used for auto focus.
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_getCameraParameters(OUT BYTE * isAutoFocus, OUT BYTE * focalLength);

/**
 * Set Camera Parameters
 *  This command is used to set the camera parameters (e.g., auto/fixed focal length as focus).
 * @param isAutoFocus: 0: fixed focus, 1: auto focus
 * @param focalLength: focal length Value 0x00-0xFF, where 0x00 is the farthest, 0xFF is nearest. Not used for auto focus.
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_setCameraParameters(IN BYTE isAutoFocus, IN BYTE focalLength);

/**
 * Get SDK Wait Time
 *
 * Get the SDK wait time for transactions
 *
 * @return SDK wait time in seconds
 *
 */
int
device_getSDKWaitTime();

/**
 * Set SDK Wait Time
 *
 * Set the SDK wait time for transactions
 *
 *
 * @param waitTime The SDK wait time for transaction in seconds
 *
 */
void
device_setSDKWaitTime(int waitTime);

/**
 * Get Thread Stack Size
 *
 * Get the stack size setting for newly created threads
 *
 * @return Thread Stack Size
 *
 */
int
device_getThreadStackSize();

/**
 * Set Thread Stack Size
 *
 * Set the stack size setting for newly created threads
 *
 *
 */
void
device_setThreadStackSize(int threadSize);

/**
 * To SD Card
 *
 * Set the destination of the file or directory function
 *
 * @param forSDCard 0: for internal memory, 1: for SD card
 *
 */
void
device_toSDCard(int forSDCard);

/**
 * Start remote key injection.
 *
 * @param caPath The path to ca-certificates.crt
 * @param isProduction 1: The reader is a production unit, 0: The reader is not a production unit
 *
 * @return success or error code.
 * @see ErrorCode
 */
int
device_startRKI(const char * caPath, int isProduction);

/**
 * Display Online Authorization Result
 *  Use this command to display the status of an online authorization request
 *  on the reader's display (OK or NOT OK).
 *  Use this command after the reader sends an online request to the issuer.
 *  The SDK timeout of the command is set to 7 seconds.
 * @param statusCode 1 = OK, 0 = NOT OK, 2 = ARC response 89 for Interac
 * @param TLV Optional TLV for AOSA
 * @param TLVLen TLV Length
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
ctls_displayOnlineAuthResult(IN int statusCode,
  IN BYTE *                         TLV,
  IN int                            TLVLen);


/**
 * Enable Pass Through
 *
 * Enables Pass Through Mode for direct communication with L1 interface
 * (power on icc, send apdu, etc).
 *
 * @param enablePassThrough 1 = pass through ON, 0 = pass through OFF
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_enablePassThrough(int enablePassThrough);

/**
 * Enable L80 Pass Through
 *
 * Enables Pass Through Mode for direct communication to L80 hook up
 * to NEO II device
 *
 * @param enableL80PassThrough 1 = pass through ON, 0 = pass through OFF
 *
 * @return RETURN_CODE:  Values can be parsed with errorCode.getErrorString()
 */
int
device_enableL80PassThrough(int enableL80PassThrough);

/**
 * Enable L100 Pass Through
 *
 * Enables Pass Through Mode for direct communication to L100 hook up
 * to NEO II device
 *
 * @param enableL100PassThrough 1 = pass through ON, 0 = pass through OFF
 *
 * @return RETURN_CODE:  Values can be parsed with errorCode.getErrorString()
 */
int
device_enableL100PassThrough(int enableL100PassThrough);

/**
 * Get L80 Pass Through Mode
 *
 * Get current Pass Through Mode for direct communication to L80
 * hook up to NEO II device
 *
 * @return RETURN_CODE: return 1 if L80 Pass Through Mode is TRUE,
 *  0 if L80 Pass Through Mode is FALSE
 */
int
device_getL80PassThroughMode();

/**
 * Get L100 Pass Through Mode
 *
 * Get current Pass Through Mode for direct communication to L100
 * hook up to NEO II device
 *
 * @return RETURN_CODE: return 1 if L100 Pass Through Mode is TRUE,
 *  0 if L100 Pass Through Mode is FALSE
 */
int
device_getL100PassThroughMode();

/**
 * Send Burst Mode
 *
 * Sets the burst mode for the device.
 *
 * @param mode 0 = OFF, 1 = Always On, 2 = Auto Exit
 *
 * @return success or error code.  Values can be parsed with
 * device_getIDGStatusCodeString
 * @see ErrorCode
 */
int
device_setBurstMode(IN BYTE mode);

/**
 * Set Poll Mode
 *
 * Sets the poll mode forthe device. Auto Poll keeps reader active,
 * Poll On Demand only polls when requested by terminal
 *
 * @param mode 0 = Auto Poll, 1 = Poll On Demand
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_setPollMode(IN BYTE mode);

/**
 * Poll for Token
 *
 * Polls for a PICC
 *
 * @param timeout timeout in milliseconds, must be multiple of 10 milliseconds. 30, 120, 630, or 1150 for example.
 *
 * @param respData Response data will be stored in respData.  1 byte of card type, and the Serial Number (or the UID) of the PICC if available.
 * @param respDataLen Length of systemCode.
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_pollForToken(IN int timeout, OUT BYTE * respData, IN_OUT int * respDataLen);

/**
 * Set Merchant Record
 * Sets the merchant record for ApplePay VAS
 *
 * @param index Merchant Record index, valid values 1-6
 * @param enabled Merchant Enabled/Valid flag
 * @param merchantID  Merchant unique identifer registered with Apple.
 *   Example com.idtechproducts.applePay
 * @param merchantURL Merchant URL, when applicable
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
device_setMerchantRecord(int index,
  int                        enabled,
  char *                     merchantID,
  char *                     merchantURL);

/**
 * DEPRECATED : please use device_getMerchantRecord_Len(IN int index,
 *                            OUT BYTE * record, IN_OUT int *recordLen)
 *
 * Get Merchant Record
 *
 * Gets the merchant record for the device.
 *
 * @param index Merchant Record index, valid values 1-6
 * @param record; needs to have at least 99 bytes of memory
 *            response data from reader.
 *            Merchant Record Index: 1 byte
 *            enabled: 1 byte
 *            Merchant ID: 32 bytes
 *            Length of Merchant URL: 1 byte
 *            Merchant URL: 64 bytes
 *
 * @return success or error code.  Values can be parsed
 *         with device_getIDGStatusCodeString()
 * @see ErrorCode
 */
int
device_getMerchantRecord(IN int index,
  OUT BYTE *                    record);

/**
 * Get Merchant Record
 *
 * Gets the merchant record for the device.
 *
 * @param index Merchant Record index, valid values 1-6
 * @param record
 *            response data from reader.
 *            Merchant Record Index: 1 byte
 *            enabled: 1 byte
 *            Merchant ID: 32 bytes
 *            Length of Merchant URL: 1 byte
 *            Merchant URL: 64 bytes
 *
 * @param recordLen Length of record
 *
 * @return success or error code.  Values can be parsed
 *         with device_getIDGStatusCodeString()
 * @see ErrorCode
 */
int
device_getMerchantRecord_Len(IN int index,
  OUT BYTE * record, IN_OUT int * recordLen);

/**
 * Get Transaction Results
 * Gets the transaction results when the reader is functioning in
 * "Auto Poll" mode
 *
 * @param cardData The transaction results
 *
 * @return success or error code. Values can be parsed
 *         with device_getIDGStatusCodeString
 * @see ErrorCode
 */
int
device_getTransactionResults(IDTMSRData * cardData);

/**
 * Start Transaction Request
 *
 * Authorizes the transaction for an MSR/CTLS/ICC card
 *
 * The tags will be returned in the callback routine.
 *
 * @param amount Transaction amount value (tag value 9F02)
 *      - SEE IMPORTANT NOTE BELOW
 * @param amtOther Other amount value, if any (tag value 9F03)
 *      - SEE IMPORTANT NOTE BELOW
 * @param type Transaction type (tag value 9C).
 * @param timeout Timeout value in seconds.
 * @param tags Any other tags to be included in the request.  Passed as TLV.
 * Example, tag 9F02 with amount 0x000000000100 would be 0x9F0206000000000100
 * If tags 9F02 (amount),9F03 (other amount), or 9C (transaction type) are
 * included, they will take priority over these values supplied as individual
 * parameters to this method.
 *
 * @param tagsLen The length of tags data buffer.
 *
 * >>>>>IMPORTANT: parameters for amount and amtOther MUST BE PASSED AS
 *    A DOUBLE VALUE WITH DECIMAL POINT.  Example, do not pass 1, but
 *    instead pass 1.0 or 1.00. Otherwise, results will be
 *    unpredictable
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 * Note: if auto poll is on, it will returm the error
 * IDG_P2_STATUS_CODE_COMMAND_NOT_ALLOWED
 *
 * NOTE ON APPLEPAY VAS:
 * To enable ApplePay VAS, first a merchant record must be defined in
 * one of the six available index positions (1-6) using
 * device_setMerchantRecord, then container tag FFEE06 must be sent as
 * part of the additional tags parameter of device_startTransaction.
 * Tag FFEE06 must contain tag 9F26 and 9F22, and can optionanally
 * contain tags 9F2B and DFO1.  Example
 * FFEE06189F220201009F2604000000009F2B050100000000DF010101
 *
 * 9F22 = two bytes = ApplePay Terminal Applicaiton Version Number.
 *        Hard defined as 0100 for now. (required)
 * 9F26 = four bytes = ApplePay Terminal Capabilities Information (required)
 *  - Byte 1 = RFU
 *  - Byte 2 = Terminal Type
 *  - - Bit 8 = VAS Support     (1=on, 0 = off)
 *  - - Bit 7 = Touch ID Required  (1=on, 0 = off)
 *  - - Bit 6 = RFU
 *  - - Bit 5 = RFU
 *  - - Bit 1,2,3,4
 *  - - - 0 = Payment Terminal
 *  - - - 1 = Transit Terminal
 *  - - - 2 = Access Terminal
 *  - - - 3 = Wireless Handoff Terminal
 *  - - - 4 = App Handoff Terminal
 *  - - - 15 = Other Terminal
 *  - Byte 3 = RFU
 *  - Byte 4 = Terminal Mode
 *  - - 0 = ApplePay VAS OR ApplePay
 *  - - 1 = ApplePay VAS AND ApplePay
 *  - - 2 = ApplePay VAS ONLY
 *  - - 3 = ApplePay ONLY
 *  9F2B = 5 bytes = ApplePay VAS Filter.  Each byte filters for that
 *         specific merchant index  (optional)
 *  DF01 = 1 byte = ApplePay VAS Protocol.  (optional)
 *  - - Bit 1 : 1 = URL VAS, 0 = Full VAS
 *  - - Bit 2 : 1 = VAS Beeps, 0 = No VAS Beeps
 *  - - Bit 3 : 1 = Silent Comm Error, 2 = EMEA Comm Error
 *  - - Bit 4-8 : RFU
 *
 */
int
device_startTransaction(IN double amount,
  IN double                       amtOther,
  IN int                          type,
  IN const int                    _timeout,
  IN BYTE *                       tags,
  IN int                          tagsLen);

/**
 * Start Loyalty Transaction Request
 *       Only for VP6800
 *
 * Authorizes the transaction for an MSR/ICC card
 *
 * The tags will be returned in the callback routine.
 *
 * @param amount Transaction amount value (tag value 9F02)
 *        - SEE IMPORTANT NOTE BELOW
 * @param amtOther Other amount value, if any (tag value 9F03)
 *        - SEE IMPORTANT NOTE BELOW
 * @param type Transaction type (tag value 9C).
 * @param timeout Timeout value in seconds.
 * @param tags Any other tags to be included in the request.  Passed
 * as TLV.  Example, tag 9F02 with amount 0x000000000100 would be
 * 0x9F0206000000000100.  If tags 9F02 (amount), 9F03 (other amount), or
 * 9C (transaction type) are included, they will take priority over
 * these values supplied as individual parameters to this method.
 *
 * @param tagsLen The length of tags data buffer.
 *
 * >>>>>IMPORTANT: parameters for amount and amtOther MUST BE PASSED
 *      AS A DOUBLE VALUE WITH DECIMAL POINT.  Example, do not pass 1,
 *      but instead pass 1.0 or 1.00. Otherwise, results will be
 *      unpredictable
 *
 *
 * @param cardType Set available card to accept. 0 = MSR only. 1 = MSR and ICC.
 *
 * @param iccReadType In case of ICC reading, this is how to behave.
 *      0 = Same as device_startTransaction
 *      1 = When reading ICC, read DF4F(JIS2EquivalentData) in ReadRecord.
 *          If the user swipes an IC card, the reader will ask for using ICC
 *      3 = When reading ICC, read DF4F(JIS2EquivalentData) in ReadRecord.
 *          If the user swipes an IC card, the reader will not ask for using ICC and output MSR data directly
 *
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 * Note: if auto poll is on, it will return the error
 * IDG_P2_STATUS_CODE_COMMAND_NOT_ALLOWED
 *
 * NOTE ON APPLEPAY VAS:
 * To enable ApplePay VAS, first a merchant record must be defined in one of
 * the six available index positions (1-6) using device_setMerchantRecord,
 * then container tag FFEE06 must be sent as part of the additional tags
 * parameter of device_startTransaction.  Tag FFEE06 must contain tag 9F26
 * and 9F22, and can optionanally contain tags 9F2B and DF01.
 * Example FFEE06189F220201009F2604000000009F2B050100000000DF010101
 * 9F22 = two bytes = ApplePay Terminal Applicaiton Version Number.
 *        Hard defined as 0100 for now. (required)
 * 9F26 = four bytes = ApplePay Terminal Capabilities Information (required)
 *  - Byte 1 = RFU
 *  - Byte 2 = Terminal Type
 *  - - Bit 8 = VAS Support    (1=on, 0 = off)
 *  - - Bit 7 = Touch ID Required  (1=on, 0 = off)
 *  - - Bit 6 = RFU
 *  - - Bit 5 = RFU
 *  - - Bit 1, 2, 3, 4
 *  - - - 0 = Payment Terminal
 *  - - - 1 = Transit Terminal
 *  - - - 2 = Access Terminal
 *  - - - 3 = Wireless Handoff Terminal
 *  - - - 4 = App Handoff Terminal
 *  - - - 15 = Other Terminal
 *  - Byte 3 = RFU
 *  - Byte 4 = Terminal Mode
 *  - - 0 = ApplePay VAS OR ApplePay
 *  - - 1 = ApplePay VAS AND ApplePay
 *  - - 2 = ApplePay VAS ONLY
 *  - - 3 = ApplePay ONLY
 *  9F2B = 5 bytes = ApplePay VAS Filter.  Each byte filters for that
 *         specific merchant index  (optional)
 *  DF01 = 1 byte = ApplePay VAS Protocol.  (optional)
 *  - - Bit 1 : 1 = URL VAS, 0 = Full VAS
 *  - - Bit 2 : 1 = VAS Beeps, 0 = No VAS Beeps
 *  - - Bit 3 : 1 = Silent Comm Error, 2 = EMEA Comm Error
 *  - - Bit 4-8 : RFU
 *
 */
int
loyalty_startTransaction(IN double amount, IN double amtOther, IN int type, IN const int _timeout, IN BYTE * tags,
  IN int tagsLen,
  IN const int cardType, IN const int iccReadType);

/**
 * Sets the transaction exponent to be used with device_startTransaction.  Default value is 2
 * @param exponent, The exponent to use when calling device_startTransaction
 */
void
device_setTransactionExponent(int exponent);

/**
 * Activate Transaction Request
 *
 * Authorizes the transaction for an MSR/CTLS/ICC card
 *
 * The tags will be returned in the callback routine.
 *
 * @param timeout Timeout value in seconds.
 * @param tags The tags to be included in the request.  Passed as a TLV.
 * Example, tag 9F02 with amount 0x000000000100 would be 0x9F0206000000000100
 * Be sure to include 9F02 (amount)and 9C (transaction type).
 *
 * @param tagsLen The length of tags data buffer.
 *
 * >>>>>IMPORTANT: parameters for amount and amtOther MUST BE PASSED AS
 *    A DOUBLE VALUE WITH DECIMAL POINT.  Example, do not pass 1, but
 *    instead pass 1.0 or 1.00. Otherwise, results will be
 *    unpredictable
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 * Note: if auto poll is on, it will returm the error
 * IDG_P2_STATUS_CODE_COMMAND_NOT_ALLOWED
 *
 * NOTE ON APPLEPAY VAS:
 * To enable ApplePay VAS, first a merchant record must be defined in
 * one of the six available index positions (1-6) using
 * device_setMerchantRecord, then container tag FFEE06 must be sent as
 * part of the additional tags parameter of device_startTransaction.
 * Tag FFEE06 must contain tag 9F26 and 9F22, and can optionanally
 * contain tags 9F2B and DFO1.  Example
 * FFEE06189F220201009F2604000000009F2B050100000000DF010101
 * 9F22 = two bytes = ApplePay Terminal Applicaiton Version Number.
 *        Hard defined as 0100 for now. (required)
 * 9F26 = four bytes = ApplePay Terminal Capabilities Information (required)
 *  - Byte 1 = RFU
 *  - Byte 2 = Terminal Type
 *  - - Bit 8 = VAS Support     (1=on, 0 = off)
 *  - - Bit 7 = Touch ID Required  (1=on, 0 = off)
 *  - - Bit 6 = RFU
 *  - - Bit 5 = RFU
 *  - - Bit 1,2,3,4
 *  - - - 0 = Payment Terminal
 *  - - - 1 = Transit Terminal
 *  - - - 2 = Access Terminal
 *  - - - 3 = Wireless Handoff Terminal
 *  - - - 4 = App Handoff Terminal
 *  - - - 15 = Other Terminal
 *  - Byte 3 = RFU
 *  - Byte 4 = Terminal Mode
 *  - - 0 = ApplePay VAS OR ApplePay
 *  - - 1 = ApplePay VAS AND ApplePay
 *  - - 2 = ApplePay VAS ONLY
 *  - - 3 = ApplePay ONLY
 *  9F2B = 5 bytes = ApplePay VAS Filter.  Each byte filters for that
 *         specific merchant index  (optional)
 *  DF01 = 1 byte = ApplePay VAS Protocol.  (optional)
 *  - - Bit 1 : 1 = URL VAS, 0 = Full VAS
 *  - - Bit 2 : 1 = VAS Beeps, 0 = No VAS Beeps
 *  - - Bit 3 : 1 = Silent Comm Error, 2 = EMEA Comm Error
 *  - - Bit 4-8 : RFU
 */
int
device_activateTransaction(IN const int _timeout,
  IN BYTE *                             tags,
  IN int                                tagsLen);

/**
 * Cancel Transaction request.
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_cancelTransaction();

/**
 * Cancel Loyalty Transaction
 *      Only for VP6800
 *
 * Cancels the currently executing transaction.
 *
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
loyalty_cancelTransaction();

/**
 * Set Cancel Transaction Mode
 *
 * Set the cancel transaction mode to be with or without LCD message
 *
 *
 * @param mode 0: With LCD message 1: Without LCD message
 * @return success or error code.  1: Success, 0: Failed
 *
 */
int
device_setCancelTransactionMode(int mode);

/**
 * Cancel Transaction Silent
 *
 * Cancel transaction with or without showing the LCD message
 *
 *
 * @param enable 0: With LCD message 1: Without LCD message
 * @return success or error code.  Values can be parsed with device_getIDGStatusCodeString
 *
 */
int
device_cancelTransactionSilent(int enable);

/**
 * Cancel Loyalty Transaction Silent
 *      Only for VP6800
 *
 * Cancel transaction with or without showing the LCD message
 *
 *
 * @param enable 0: With LCD message 1: Without LCD message
 * @return success or error code.  Values can be parsed with device_getIDGStatusCodeString
 *
 */
int
loyalty_cancelTransactionSilent(int enable);

/**
 * Configures the buttons on the ViVOpay Vendi reader
 * @param done
 *   0x01: the Done switch is enabled
 *   0x00: the Done switch is disabled
 *
 * @param swipe
 *   0x01: the Swipe Card switch is enabled
 *   0x00: the Swipe Card switch is disabled
 *
 * @param delay
 *   an unsigned delay value (<= 30) in seconds
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_configureButtons(IN BYTE done,
  IN BYTE                       swipe,
  IN BYTE                       delay);

/**
 * Reads the button configuration from the ViVOpay Vendi reader
 * @param done
 *     0x01: the Done switch is enabled
 *     0x00: the Done switch is disabled
 *
 * @param swipe
 *     0x01: the Swipe Card switch is enabled
 *     0x00: the Swipe Card switch is disabled
 *
 * @param delay
 *     an unsigned delay value in seconds
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_getButtonConfiguration(OUT BYTE * done,
  OUT BYTE *                             swipe,
  OUT BYTE *                             delay);

/**
 * Stops the blue LEDs on the ViVOpay Vendi reader from flashing in
 *   left to right sequence and turns the LEDs off, and contactless
 *   function is disabled at the same time
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */

int
device_disableBlueLED();

/**
 * Use this function to control the blue LED behavior on the Vendi reader
 *
 * @param data
 *     Sequence data
 *       Byte 0 (Cycle): 0 = Cycle once, 1 = Repeat
 *       Byte 1 (LEDs): LED State Bitmap
 *       Byte 2-3 (Duration): Given in multiples of 10 millisecond
 *       Byte 4 (LED): LED State Bitmap
 *       Byte 5-6 (Duration): Given in multiples of 10 millisecond
 *       Byte 7-24 (Additional LED/Durations): Define up to 8 LED and
 *                 duration pars
 *
 *     LED State Bitmap:
 *             Bit 8: Left blue LED, 0 = off, 1 = on
 *             Bit 7: Center Blue LED, 0 = off, 1 = on
 *             Bit 6: Right Blue LED0 = off, 1 = on
 *             Bit 5: Yellow LED, 0 = off, 1 = on
 *             Bit 4: Reserved for future use
 *             Bit 3: Reserved for future use
 *             Bit 2: Reserved for future use
 *             Bit 1: Reserved for future use
 *
 * @param dataLen
 *     Length of the sequence data: 0 or 4 to 25 bytes
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_enableBlueLED(IN BYTE * data,
  IN int                       dataLen);

/**
 * Use this function to clear the LCD display
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_lcdDisplayClear();

/**
 * Enable or disable the external LCD message It will turn off the
 *   external LCD messages including EMV transactions.  (For the
 *   users who only need MSR and/or CTLS transactions.)  The function
 *   only works for VP5300
 *
 *    @param enableExtLCDMsg  1 = ON, 0 = OFF
 *    @return success or error code.  Values can be parsed
 *    with device_getIDGStatusCodeString
 *    @see ErrorCode
 */
int
device_enableExternalLCDMessages(IN int enableExtLCDMsg);

/**
 * Enable or disable the RF Antenna
 *
 *
 *    @param enableAntenna  1 = ON, 0 = OFF
 *    @return success or error code.  Values can be parsed with device_getIDGStatusCodeString
 *    @see ErrorCode
 */
int
device_enableRFAntenna(IN int enableAntenna);

/**
 * Use this function to turn off the ViVOpay Vendi reader yellow
 * LED. This LED is located below the three blue LEDs
 *
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_turnOffYellowLED();

/**
 * Use this function to turn on the ViVOpay Vendi reader yellow
 * LED. This LED is located below the three blue LEDs
 *
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_turnOnYellowLED();

/**
 * Use this function to make the buzzer beep once
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_buzzerOnOff();

/**
 * Use this function to display text on the LCD display.
 *      On the Vendi reader the LCD is a 2-line character display.
 *
 * @param message
 *      Valid messages for the first line of text are between 1 and 16
 *      printable characters long.
 *      If the text message is greater than 16 bytes but not more than 32 bytes,
 *           byte 17 and onward are displayed as a second row of text.
 *      All messages are left justified on the LCD display.
 *
 * @param messageLen
 *      Length of the message: 1 to 32 bytes
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_lcdDisplayLine1Message(IN BYTE * message,
  IN int                                messageLen);

/**
 * Use this function to display the message on line 2 of the LCD display.
 *      On the Vendi reader the LCD is a 2-line character display.
 *
 * @param message
 *      Valid messages are between 1 and 16 printable characters long.
 *      All messages are left justified on the LCD display.
 *
 * @param messageLen
 *      Length of the message: 1 to 16 bytes
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_lcdDisplayLine2Message(IN BYTE * message,
  IN int                                messageLen);

/**
 * Get Key Status
 *
 * Gets the status of loaded keys
 * @param status newFormat for Augusta and miniSmartII only
 *     1: new format of key status
 *     0: reserved format for support previous device
 * @param status
 *     For L80, L100, Augusta and miniSmartII:
 *        When the newFormat is 0, data format as follows.
 *        For Augusta and miniSmartII:
 *          byte 0:  PIN DUKPT Key, Does not support, always 0
 *          byte 1:  PIN Master Key, Does not support, always 0
 *          byte 2:  PIN Session Key, Does not support, always 0
 *          byte 3:  Account/MSR DUKPT Key, 1 Exists, 0 None, 0xFF STOP
 *          byte 4:  Account/ICC DUKPT Key, 1 Exists, 0 None, 0xFF STOP
 *          byte 5:  Admin DUKPT Key, 1 Exists, 0 None, 0xFF STOP
 *        For L80 and L100:
 *          byte 0:  PIN DUKPT Key
 *          byte 1:  PIN Master Key
 *          byte 2:  Standard PIN Session Key
 *          byte 3:  Desjardins PIN Session Key
 *          byte 4:  Account/MSR DUKPT Key, 1 Exists, 0 None, 0xFF STOP, Does not support, always 0
 *          byte 5:  Account/ICC DUKPT Key, 1 Exists, 0 None, 0xFF STOP, Does not support, always 0
 *          byte 6:  Admin DUKPT Key, 1 Exists, 0 None, 0xFF STOP
 *          byte 7:  Data DUKPT Key, 1 Exists, 0 None, 0xFF STOP
 *          byte 8:  MAC DUKPT Key, 1 Exists, 0 None, 0xFF STOP
 *
 *        when the newFormat is 1, data format as follows.
 *         [Block Length] [KeyStatusBlock1] [KeyStatusBlock2]...[KeyStatusBlockN]
 *        Where:
 *        [Block Length] is 2 bytes, format is Len_L Len_H, is KeyStatusBlock Number
 *        [KeyStatusBlockX> is 4 bytes, format is [Key Index and Key Name] [key slot] [key status]:
 *        [Key Index and Key Name] is 1 byte. Please refer to following table
 *            0x14    LCL-KEK to Encrypt Other Keys
 *            0x02    Data encryption Key to Encrypt ICC/MSR
 *            0x05    MAC DUKPT Key for Host-Device - MAC Verification
 *            0x05    MTK DUKPT Key for TTK Self-Test
 *            0x0C    RKI-KEK for Remote Key Injection
 *        [key slot] is 2 bytes. Range is 0 - 9999
 *        the MTK DUKPT Key slot is 16, the others are all 0
 *        [key status] is 1 byte.
 *            0 - Not Exist
 *            1 - Exist
 *        0xFF - (Stop. Only Valid for DUKPT Key)
 *    For NEO2 and SREDKey2:
 *		Each unit of three bytes represents one key's parameters (index and slot).
 *			Key Name Index (1 byte):
 *				0x14 - LCL-KEK
 *				0x01 - Pin encryption Key (NEO2 only)
 *				0x02 - Data encryption Key
 *				0x05 - MAC DUKPT Key
 *				0x0A - PCI Pairing Key (NEO2 only)
 *			Key Slot (2 bytes):
 *				Indicate different slots of a certain Key Name
 *					Example: slot =5 (0x00 0x05), slot=300 (0x01 0x2C)
 *					For BTPay380, slot is always 0
 *		For example, 0x14 0x00 0x00 0x02 0x00 0x00 0x0A 0x00 0x00 will represent
 *			[KeyNameIndex=0x14,KeySlot=0x0000], [KeyNameIndex=0x02,KeySlot=0x0000] and [KeyNameIndex=0x0A,KeySlot=0x0000]
 *
 * @param statusLen the length of status
 *
 * @return RETURN_CODE:    Values can be parsed with device_getIDGStatusCodeString
 */
int
device_getKeyStatus(int * newFormat, BYTE * status, int * statusLen);

/**
 * Update Firmware
 * Updates the firmware of NEO 2 devices.
 * @param firmwareData Signed binary data of a firmware file provided by IDTech
 * @param firmwareDataLen Length of firmwareData
 * @param firmwareName Firmware name.
 *  - For example "VP5300_v1.00.023.0167.S_Test.fm"
 * @param encryptionType Encryption type
 * - 0 : Plaintext
 * - 1 : TDES ECB, PKCS#5 padding
 * - 2 : TDES CBC, PKCS#5, IV is all 0
 * @param keyBlob Encrypted firmware session key blob, TR-31 Rev B, wrapped by
 *  FW Key (Optional, none if firmware is plaintext)
 * @param keyBlobLen Length of keyBlob
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 * Firmware update status is returned in the callback with the following values:
 * sender = device type
 * state = DEVICE_FIRMWARE_UPDATE
 * current block
 * total blocks
 * ResultCode:
 * - RETURN_CODE_DO_SUCCESS = Firmware Update Started Successfully
 * - Any other return code represents an error condition
 *
 */
int
device_updateFirmware(IN BYTE * firmwareData,
  IN int                        firmwareDataLen,
  IN char *                     firmwareName,
  IN int                        encryptionType,
  IN BYTE *                     keyBlob,
  IN int                        keyBlobLen);

/**
 * Update Firmware with zip file
 * Updates the firmware of NEO 2 devices.
 * @param firmwareZipFilename Firmware zip file name.
 *  - For example "package_VP6300 FW v1.01.003.0432.T.zip"
 * @param firmwareZipFilenameLen Firmware zip file name length
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 * Firmware update status is returned in the callback with the following values:
 * sender = device type
 * state = DEVICE_FIRMWARE_UPDATE
 * current block
 * total blocks
 * ResultCode:
 * - RETURN_CODE_DO_SUCCESS = Firmware Update Completed Successfully
 * - Any other return code represents an error condition
 * Note: to call this function under Windows, the executable file "unzip.exe" is required
 *
 */
//int
//device_updateFirmwareWithZipFile(IN char *firmwareZipFilename, IN int firmwareZipFilenameLen);

/**
 * Transfer File
 *  This command transfers a data file to the reader.
 * @param fileName Filename.  The data for this command is a ASCII string with the
 *   complete path and file name you want to create. You do not need to
 *   specify the root directory. Indicate subdirectories with a forward slash (/).
 * @param filenameLen File Name Length.
 * @param file The data file.
 * @param fileLen File Length.
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_transferFile(IN char * fileName, IN int fileNameLen, IN BYTE * file, IN int fileLen);

/**
 * Delete File
 *  This command deletes a file or group of files.
 * @param filename Complete path and file name of the file you want to delete.
 *   You do not need to specify the root directory. Indicate subdirectories with a forward slash (/).
 * @param filenameLen File Name Length.
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_deleteFile(IN char * fileName, IN int fileNameLen);

/**
 * Query File
 *  This command checks if the specified file exists in NAND Flash..
 * @param directoryName Directory name string.  No longer than 32 bytes.  ASCII string, terminated by 0x00.
 * @param directoryNameLen Directory Name Length.
 * @param fileName File name string.  No longer than 32 bytes.  ASCII string, terminated by 0x00.
 * @param fileNameLen File Name Length.
 * @param isExist File exists: 1, File not exists 0.
 * @param timeStamp Latest time stamp of the file.  6 bytes BCD code if the file exists.
 * @param timeStampLen Length of timeStamp.  6 if the file exists, 0 if the file does not exist.
 * @param fileSize Zero-terminated ASCII string of the file size.
 * @param fileSizeLen Length of filesSize.
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_queryFile(IN char * directoryName, IN int directoryNameLen, IN char * fileName, IN int fileNameLen,
  OUT int * isExist, OUT BYTE * timeStamp, IN_OUT int * timeStampLen, OUT char * fileSize, IN_OUT int * fileSizeLen);

/**
 * Read Out File from SD Card
 *  This command retrieves a file from the SD card of the reader.
 * @param directoryName Directory name string.  No longer than 32 bytes.  ASCII string, terminated by 0x00.
 * @param directoryNameLen Directory Name Length.
 * @param fileName File name string.  No longer than 32 bytes.  ASCII string, terminated by 0x00.
 * @param fileNameLen File Name Length.
 * @param startingOffset Starting offset in the file to retrieve
 * @param numBytes Number of bytes of file data to retrieve
 * @param fileData  the file  data read from the SD card
 * @param fileDataLen Length of fileData.
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_readFileFromSD(IN char * directoryName, IN int directoryNameLen, IN char * fileName, IN int fileNameLen,
  IN int startingOffset, IN int numBytes, OUT BYTE * fileData, IN_OUT int * fileDataLen);

/**
 * Start Listen Notifications
 *  This function enables Card Status and Front Switch notifications.
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_startListenNotifications();

/**
 * Stop Listen Notifications
 *  This function disables Card Status and Front Switch notifications.
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_stopListenNotifications();

/**
 * Start QR Code Scanning
 *
 * Enables QR Code scanning with the default window, waiting for the QR code.
 * @param timeout QR Code Scan Timeout Value.  Between 30 and 65536 seconds.
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 * Note: if auto poll mode is on, it will return command not allowed error
 */
int
device_startQRCodeScan(IN int _timeout);

/**
 * Start QR Code Scanning with Display Window Info
 *
 * Enables QR Code scanning, waiting for the QR code.
 * @param timeout QR Code Scan Timeout Value.  Between 30 and 65536 seconds.
 * @param x the x-coordinate of the display window.
 * @param y the y-coordinate of the display window.
 * @param width the width of the display window.
 * @param height the height of the display window.
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 * Note: if auto poll mode is on, it will return command not allowed error
 */
int
device_startQRCodeScanWithDisplayWindowInfo(IN int _timeout, IN int x, IN int y, IN int width, IN int height);

/**
 * Stop QR Code Scanning
 * Cancels QR Code scanning request.
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_stopQRCodeScan();

/**
 * Start Taking Photo
 *
 * Enables the camera to take a photo.
 * @param timeout Photo taking Timeout Value.  Between 30 and 65536 seconds.
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 * Note: if auto poll mode is on, it will return command not allowed error
 */
int
device_startTakingPhoto(IN int _timeout);

/**
 * Stop Taking Photo
 * Cancels Photo Taking request.
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_stopTakingPhoto();

/**
 * Review the return code description.<br/>
 * @param returnCode  the response result.
 * @param description
 * @retval the string for description of response result
 *
 * - 0: "no error, beginning task";
 * - 1: "no response from reader";
 * - 2: "invalid response data";
 * - 3: "time out for task or CMD";
 * - 4: "wrong parameter";
 * - 5: "SDK is doing MSR or ICC task";
 * - 6: "SDK is doing PINPad task";
 * - 7: "SDK is doing CTLS task";
 * - 8: "SDK is doing EMV task";
 * - 9: "SDK is doing Other task";
 * - 10: "err response or data";
 * - 11: "no reader attached";
 * - 12: "mono audio is enabled";
 * - 13: "did connection";
 * - 14: "audio volume is too low";
 * - 15: "task or CMD be canceled";
 * - 16: "UF wrong string format";
 * - 17: "UF file not found";
 * - 18: "UF wrong file format";
 * - 19: "Attempt to contact online host failed";
 * - 20: "Attempt to perform RKI failed";
 * - 22: "Buffer size is not enough";
 * - 0X300: "Key Type(TDES) of Session Key is not same as the related
 *           Master Key.";
 * - 0X400: "Related Key was not loaded.";
 * - 0X500: "Key Same.";
 * - 0X501: "Key is all zero";
 * - 0X502: "TR-31 format error";
 * - 0X702: "PAN is Error Key.";
 * - 0X705: "No Internal MSR PAN (or Internal MSR PAN is erased timeout)";
 * - 0X0C01: "Incorrect Frame Tag";
 * - 0X0C02: "Incorrect Frame Type";
 * - 0X0C03: "Unknown Frame Type";
 * - 0X0C04: "Unknown Command";
 * - 0X0C05: "Unknown Sub-Command";
 * - 0X0C06: "CRC Error";
 * - 0X0C07: "Failed";
 * - 0X0C08: "Timeout";
 * - 0X0C0A: "Incorrect Parameter";
 * - 0X0C0B: "Command Not Supported";
 * - 0X0C0C: "Sub-Command Not Supported";
 * - 0X0C0D: "Parameter Not Supported / Status Abort Command";
 * - 0X0C0F: "Sub-Command Not Allowed";
 * - 0X0D01: "Incorrect Header Tag";
 * - 0X0D02: "Unknown Command";
 * - 0X0D03: "Unknown Sub-Command";
 * - 0X0D04: "CRC Error in Frame";
 * - 0X0D05: "Incorrect Parameter";
 * - 0X0D06: "Parameter Not Supported";
 * - 0X0D07: "Mal-formatted Data";
 * - 0X0D08: "Timeout";
 * - 0X0D0A: "Failed / NACK";
 * - 0X0D0B: "Command not Allowed";
 * - 0X0D0C: "Sub-Command not Allowed";
 * - 0X0D0D: "Buffer Overflow (Data Length too large for reader buffer)";
 * - 0X0D0E: "User Interface Event";
 * - 0X0D11: "Communication type not supported, VT-1, burst, etc.";
 * - 0X0D12: "Secure interface is not functional or is in an intermediate state.";
 * - 0X0D13: "Data field is not mod 8";
 * - 0X0D14: "Pad  - 0X80 not found where expected";
 * - 0X0D15: "Specified key type is invalid";
 * - 0X0D1: "Could not retrieve key from the SAM(InitSecureComm)";
 * - 0X0D17: "Hash code problem";
 * - 0X0D18: "Could not store the key into the SAM(InstallKey)";
 * - 0X0D19: "Frame is too large";
 * - 0X0D1A: "Unit powered up in authentication state but POS must resend the InitSecureComm command";
 * - 0X0D1B: "The EEPROM may not be initialized because SecCommInterface does not make sense";
 * - 0X0D1C: "Problem encoding APDU";
 * - 0X0D20: "Unsupported Index(ILM) SAM Transceiver error - problem communicating with the SAM(Key Mgr)";
 * - 0X0D2: "Unexpected Sequence Counter in multiple frames for single bitmap(ILM) Length error in data returned from the SAM(Key Mgr)";
 * - 0X0D22: "Improper bit map(ILM)";
 * - 0X0D23: "Request Online Authorization";
 * - 0X0D24: "ViVOCard3 raw data read successful";
 * - 0X0D25: "Message index not available(ILM) ViVOcomm activate transaction card type(ViVOcomm)";
 * - 0X0D26: "Version Information Mismatch(ILM)";
 * - 0X0D27: "Not sending commands in correct index message index(ILM)";
 * - 0X0D28: "Time out or next expected message not received(ILM)";
 * - 0X0D29: "ILM languages not available for viewing(ILM)";
 * - 0X0D2A: "Other language not supported(ILM)";
 * - 0X0D41: "Unknown Error from SAM";
 * - 0X0D42: "Invalid data detected by SAM";
 * - 0X0D43: "Incomplete data detected by SAM";
 * - 0X0D44: "Reserved";
 * - 0X0D45: "Invalid key hash algorithm";
 * - 0X0D46: "Invalid key encryption algorithm";
 * - 0X0D47: "Invalid modulus length";
 * - 0X0D48: "Invalid exponent";
 * - 0X0D49: "Key already exists";
 * - 0X0D4A: "No space for new RID";
 * - 0X0D4B: "Key not found";
 * - 0X0D4C: "Crypto not responding";
 * - 0X0D4D: "Crypto communication error";
 * - 0X0D4E: "Module-specific error for Key Manager";
 * - 0X0D4F: "All key slots are full (maximum number of keys has been installed)";
 * - 0X0D50: "Auto-Switch OK";
 * - 0X0D51: "Auto-Switch failed";
 * - 0X0D90: "Account DUKPT Key not exist";
 * - 0X0D91: "Account DUKPT Key KSN exausted";
 * - 0X0D00: "This Key had been loaded.";
 * - 0X0E00: "Base Time was loaded.";
 * - 0X0F00: "Encryption Or Decryption Failed.";
 * - 0X1000: "Battery Low Warning (It is High Priority Response while Battery is Low.)";
 * - 0X1800: "Send 'Cancel Command' after send 'Get Encrypted PIN' & 'Get Numeric' & 'Get Amount';
 * - 0X1900: "Press 'Cancel' key after send 'Get Encrypted PIN' & 'Get Numeric' & 'Get Amount';
 * - 0X30FF: "Security Chip is not connect";
 * - 0X3000: "Security Chip is deactivation & Device is In Removal Legally State.";
 * - 0X3101: "Security Chip is activation &  Device is In Removal Legally State.";
 * - 0X5500: "No Admin DUKPT Key.";
 * - 0X5501: "Admin  DUKPT Key STOP.";
 * - 0X5502: "Admin DUKPT Key KSN is Error.";
 * - 0X5503: "Get Authentication Code1 Failed.";
 * - 0X5504: "Validate Authentication Code Error.";
 * - 0X5505: "Encrypt or Decrypt data failed.";
 * - 0X5506: "Not Support the New Key Type.";
 * - 0X5507: "New Key Index is Error.";
 * - 0X5508: "Step Error.";
 * - 0X5509: "KSN Error";
 * - 0X550A: "MAC Error.";
 * - 0X550B: "Key Usage Error.";
 * - 0X550C: "Mode Of Use Error.";
 * - 0X550F: "Other Error.";
 * - 0X6000: "Save or Config Failed / Or Read Config Error.";
 * - 0X6200: "No Serial Number.";
 * - 0X6900: "Invalid Command - Protocol is right, but task ID is invalid.";
 * - 0X6A01: "Unsupported Command - Protocol and task ID are right, but command is invalid - In this State";
 * - 0X6A00: "Unsupported Command - Protocol and task ID are right, but command is invalid.";
 * - 0X6B00: "Unknown parameter in command - Protocol task ID and command are right, but parameter is invalid.";
 * - 0X6C00: "Unknown parameter in command - Protocol task ID and command are right, but length is out of the requirement.";
 * - 0X7200: "Device is suspend (MKSK suspend or press password suspend).";
 * - 0X7300: "PIN DUKPT is STOP (21 bit 1).";
 * - 0X7400: "Device is Busy.";
 * - 0XE100: "Can not enter sleep mode";
 * - 0XE200: "File has existed";
 * - 0XE300: "File has not existed";
 * - 0XE313: "IO line low -- Card error after session start";
 * - 0XE400: "Open File Error";
 * - 0XE500: "SmartCard Error";
 * - 0XE600: "Get MSR Card data is error";
 * - 0XE700: "Command time out";
 * - 0XE800: "File read or write is error";
 * - 0XE900: "Active 1850 error!";
 * - 0XEA00: "Load bootloader error";
 * - 0XEF00: "Protocol Error- STX or ETX or check error.";
 * - 0XEB00: "Picture is not exist";
 * - 0X2C02: "No Microprocessor ICC seated";
 * - 0X2C06: "no card seated to request ATR";
 * - 0X2D01: "Card Not Supported, ";
 * - 0X2D03: "Card Not Supported, wants CRC";
 * - 0X690D: "Command not supported on reader without ICC support";
 * - 0X8100: "ICC error time out on power-up";
 * - 0X8200: "invalid TS character received - Wrong operation step";
 * - 0X8300: "Decode MSR Error";
 * - 0X8400: "TriMagII no Response";
 * - 0X8500: "No Swipe MSR Card";
 * - 0X8510: "No Financial Card";
 * - 0X8600: "Unsupported F, D, or combination of F and D";
 * - 0X8700: "protocol not supported EMV TD1 out of range";
 * - 0X8800: "power not at proper level";
 * - 0X8900: "ATR length too long";
 * - 0X8B01: "EMV invalid TA1 byte value";
 * - 0X8B02: "EMV TB1 required";
 * - 0X8B03: "EMV Unsupported TB1 only 00 allowed";
 * - 0X8B04: "EMV Card Error, invalid BWI or CWI";
 * - 0X8B06: "EMV TB2 not allowed in ATR";
 * - 0X8B07: "EMV TC2 out of range";
 * - 0X8B08: "EMV TC2 out of range";
 * - 0X8B09: "per EMV96 TA3 must be >  - 0XF";
 * - 0X8B10: "ICC error on power-up";
 * - 0X8B11: "EMV T=1 then TB3 required";
 * - 0X8B12: "Card Error, invalid BWI or CWI";
 * - 0X8B13: "Card Error, invalid BWI or CWI";
 * - 0X8B17: "EMV TC1/TB3 conflict-";
 * - 0X8B20: "EMV TD2 out of range must be T=1";
 * - 0X8C00: "TCK error";
 * - 0XA304: "connector has no voltage setting";
 * - 0XA305: "ICC error on power-up invalid (SBLK(IFSD) exchange";
 * - 0XE301: "ICC error after session start";
 * - 0XFF00: "Request to go online";
 * - 0XFF01: "EMV: Accept the offline transaction";
 * - 0XFF02: "EMV: Decline the offline transaction";
 * - 0XFF03: "EMV: Accept the online transaction";
 * - 0XFF04: "EMV: Decline the online transaction";
 * - 0XFF05: "EMV: Application may fallback to magstripe technology";
 * - 0XFF06: "EMV: ICC detected tah the conditions of use are not satisfied";
 * - 0XFF07: "EMV: ICC didn't accept transaction";
 * - 0XFF08: "EMV: Transaction was cancelled";
 * - 0XFF09: "EMV: Application was not selected by kernel or ICC format error or ICC missing data error";
 * - 0XFF0A: "EMV: Transaction is terminated";
 * - 0XFF0B: "EMV: Other EMV Error";
 * - 0XFFFF: "NO RESPONSE";
 * - 0XF002: "ICC communication timeout";
 * - 0XF003: "ICC communication Error";
 * - 0XF00F: "ICC Card Seated and Highest Priority, disable MSR work request";
 * - 0XF200: "AID List / Application Data is not exist";
 * - 0XF201: "Terminal Data is not exist";
 * - 0XF202: "TLV format is error";
 * - 0XF203: "AID List is full";
 * - 0XF204: "Any CA Key is not exist";
 * - 0XF205: "CA Key RID is not exist";
 * - 0XF206: "CA Key Index it not exist";
 * - 0XF207: "CA Key is full";
 * - 0XF208: "CA Key Hash Value is Error";
 * - 0XF209: "Transaction  format error";
 * - 0XF20A: "The command will not be processing";
 * - 0XF20B: "CRL is not exist";
 * - 0XF20C: "CRL number  exceed max number";
 * - 0XF20D: "Amount, Other Amount, Trasaction Type  are  missing";
 * - 0XF20E: "The Identification of algorithm is mistake";
 * - 0XF20F: "No Financial Card";
 * - 0XF210: "In Encrypt Result state, TLV total Length is greater than Max Length";
 * - 0X1001: "INVALID ARG";
 * - 0X1002: "FILE_OPEN_FAILED";
 * - 0X1003: "FILE OPERATION_FAILED";
 * - 0X2001: "MEMORY_NOT_ENOUGH";
 * - 0X3002: "SMARTCARD_FAIL";
 * - 0X3003: "SMARTCARD_INIT_FAILED";
 * - 0X3004: "FALLBACK_SITUATION";
 * - 0X3005: "SMARTCARD_ABSENT";
 * - 0X3006: "SMARTCARD_TIMEOUT";
 * - 0X3012: "EMV_RESULT_CODE_MSR_CARD_ERROR_FALLBACK";
 * - 0X5001: "EMV_PARSING_TAGS_FAILED";
 * - 0X5002: "EMV_DUPLICATE_CARD_DATA_ELEMENT";
 * - 0X5003: "EMV_DATA_FORMAT_INCORRECT";
 * - 0X5004: "EMV_NO_TERM_APP";
 * - 0X5005: "EMV_NO_MATCHING_APP";
 * - 0X5006: "EMV_MISSING_MANDATORY_OBJECT";
 * - 0X5007: "EMV_APP_SELECTION_RETRY";
 * - 0X5008: "EMV_GET_AMOUNT_ERROR";
 * - 0X5009: "EMV_CARD_REJECTED";
 * - 0X5010: "EMV_AIP_NOT_RECEIVED";
 * - 0X5011: "EMV_AFL_NOT_RECEIVED";
 * - 0X5012: "EMV_AFL_LEN_OUT_OF_RANGE";
 * - 0X5013: "EMV_SFI_OUT_OF_RANGE";
 * - 0X5014: "EMV_AFL_INCORRECT";
 * - 0X5015: "EMV_EXP_DATE_INCORRECT";
 * - 0X5016: "EMV_EFF_DATE_INCORRECT";
 * - 0X5017: "EMV_ISS_COD_TBL_OUT_OF_RANGE";
 * - 0X5018: "EMV_CRYPTOGRAM_TYPE_INCORRECT";
 * - 0X5019: "EMV_PSE_NOT_SUPPORTED_BY_CARD";
 * - 0X5020: "EMV_USER_SELECTED_LANGUAGE";
 * - 0X5021: "EMV_SERVICE_NOT_ALLOWED";
 * - 0X5022: "EMV_NO_TAG_FOUND";
 * - 0X5023: "EMV_CARD_BLOCKED";
 * - 0X5024: "EMV_LEN_INCORRECT";
 * - 0X5025: "CARD_COM_ERROR";
 * - 0X5026: "EMV_TSC_NOT_INCREASED";
 * - 0X5027: "EMV_HASH_INCORRECT";
 * - 0X5028: "EMV_NO_ARC";
 * - 0X5029: "EMV_INVALID_ARC";
 * - 0X5030: "EMV_NO_ONLINE_COMM";
 * - 0X5031: "TRAN_TYPE_INCORRECT";
 * - 0X5032: "EMV_APP_NO_SUPPORT";
 * - 0X5033: "EMV_APP_NOT_SELECT";
 * - 0X5034: "EMV_LANG_NOT_SELECT";
 * - 0X5035: "EMV_NO_TERM_DATA";
 * - 0X5039: "EMV_PIN_ENTRY_TIMEOUT";
 * - 0X6001: "CVM_TYPE_UNKNOWN";
 * - 0X6002: "CVM_AIP_NOT_SUPPORTED";
 * - 0X6003: "CVM_TAG_8E_MISSING";
 * - 0X6004: "CVM_TAG_8E_FORMAT_ERROR";
 * - 0X6005: "CVM_CODE_IS_NOT_SUPPORTED";
 * - 0X6006: "CVM_COND_CODE_IS_NOT_SUPPORTED";
 * - 0X6007: "NO_MORE_CVM";
 * - 0X6008: "PIN_BYPASSED_BEFORE";
 * - 0X7001: "PK_BUFFER_SIZE_TOO_BIG";
 * - 0X7002: "PK_FILE_WRITE_ERROR";
 * - 0X7003: "PK_HASH_ERROR";
 * - 0X8001: "NO_CARD_HOLDER_CONFIRMATION";
 * - 0X8002: "GET_ONLINE_PIN";
 * - 0XD000: "Data not exist";
 * - 0XD001: "Data access error";
 * - 0XD100: "RID not exist";
 * - 0XD101: "RID existed";
 * - 0XD102: "Index not exist";
 * - 0XD200: "Maximum exceeded";
 * - 0XD201: "Hash error";
 * - 0XD205: "System Busy";
 * - 0X0E01: "Unable to go online";
 * - 0X0E02: "Technical Issue";
 * - 0X0E03: "Declined";
 * - 0X0E04: "Issuer Referral transaction";
 * - 0X0F01: "Decline the online transaction";
 * - 0X0F02: "Request to go online";
 * - 0X0F03: "Transaction is terminated";
 * - 0X0F05: "Application was not selected by kernel or ICC format error or ICC missing data error";
 * - 0X0F07: "ICC didn't accept transaction";
 * - 0X0F0A: "Application may fallback to magstripe technology";
 * - 0X0F0C: "Transaction was cancelled";
 * - 0X0F0D: "Timeout";
 * - 0X0F0F: "Other EMV Error";
 * - 0X0F10: "Accept the offline transaction";
 * - 0X0F11: "Decline the offline transaction";
 * - 0X0F21: "ICC detected tah the conditions of use are not satisfied";
 * - 0X0F22: "No app were found on card matching terminal configuration";
 * - 0X0F23: "Terminal file does not exist";
 * - 0X0F24: "CAPK file does not exist";
 * - 0X0F25: "CRL Entry does not exist";
 * - 0X0FFE: "code when blocking is disabled";
 * - 0X0FFF: "code when command is not applicable on the selected device";
 * - 0XF005: "ICC Encrypted C-APDU Data Structure Length Error Or Format Error.";
 * - 0XBBE0: "CM100 Success";
 * - 0XBBE1: "CM100 Parameter Error";
 * - 0XBBE2: "CM100 Low Output Buffer";
 * - 0XBBE3: "CM100 Card Not Found";
 * - 0XBBE4: "CM100 Collision Card Exists";
 * - 0XBBE5: "CM100 Too Many Cards Exist";
 * - 0XBBE6: "CM100 Saved Data Does Not Exist";
 * - 0XBBE8: "CM100 No Data Available";
 * - 0XBBE9: "CM100 Invalid CID Returned";
 * - 0XBBEA: "CM100 Invalid Card Exists";
 * - 0XBBEC: "CM100 Command Unsupported";
 * - 0XBBED: "CM100 Error In Command Process";
 * - 0XBBEE: "CM100 Invalid Command";
 *
 * - 0X9031: "Unknown command";
 * - 0X9032: "Wrong parameter (such as the length of the command is incorrect)";
 *
 * - 0X9038: "Wait (the command couldnt be finished in BWT)";
 * - 0X9039: "Busy (a previously command has not been finished)";
 * - 0X903A: "Number of retries over limit";
 *
 * - 0X9040: "Invalid Manufacturing system data";
 * - 0X9041: "Not authenticated";
 * - 0X9042: "Invalid Master DUKPT Key";
 * - 0X9043: "Invalid MAC Key";
 * - 0X9044: "Reserved for future use";
 * - 0X9045: "Reserved for future use";
 * - 0X9046: "Invalid DATA DUKPT Key";
 * - 0X9047: "Invalid PIN Pairing DUKPT Key";
 * - 0X9048: "Invalid DATA Pairing DUKPT Key";
 * - 0X9049: "No nonce generated";
 * - 0X9949: "No GUID available.  Perform getVersion first.";
 * - 0X9950: "MAC Calculation unsuccessful. Check BDK value.";
 *
 * - 0X904A: "Not ready";
 * - 0X904B: "Not MAC data";
 *
 * - 0X9050: "Invalid Certificate";
 * - 0X9051: "Duplicate key detected";
 * - 0X9052: "AT checks failed";
 * - 0X9053: "TR34 checks failed";
 * - 0X9054: "TR31 checks failed";
 * - 0X9055: "MAC checks failed";
 * - 0X9056: "Firmware download failed";
 *
 * - 0X9060: "Log is full";
 * - 0X9061: "Removal sensor unengaged";
 * - 0X9062: "Any hardware problems";
 *
 * - 0X9070: "ICC communication timeout";
 * - 0X9071: "ICC data error (such check sum error)";
 * - 0X9072: "Smart Card not powered up";
 */
void
device_getResponseCodeString(IN int returnCode, OUT char * despcrition);

/**
 * Review the return code description.<br/>
 * @param returnCode  the response result.
 * @param description
 * @retval the string for description of response result
 *
 * - 0: "no error, beginning task";
 * - 1: "no response from reader";
 * - 2: "invalid response data";
 *
 * - 01: " Incorrect Header Tag";
 * - 02: " Unknown Command";
 * - 03: " Unknown Sub-Command";
 * - 04: " CRC Error in Frame";
 * - 05: " Incorrect Parameter";
 * - 06: " Parameter Not Supported";
 * - 07: " Mal-formatted Data";
 * - 08: " Timeout";
 * - 0A: " Failed / NACK";
 * - 0B: " Command not Allowed";
 * - 0C: " Sub-Command not Allowed";
 * - 0D: " Buffer Overflow (Data Length too large for reader buffer)";
 * - 0E: " User Interface Event";
 * - 10: " Need clear firmware(apply in boot loader only)";
 * - 11: " Communication type not supported, VT-1, burst, etc. Need encrypted firmware (apply in boot loader only)";
 * - 12: " Secure interface is not functional or is in an intermediate state.";
 * - 13: " Data field is not mod 8";
 * - 14: " Pad 0x80 not found where expected";
 * - 15: " Specified key type is invalid";
 * - 16: " Could not retrieve key from the SAM (InitSecureComm)";
 * - 17: " Hash code problem";
 * - 18: " Could not store the key into the SAM (InstallKey)";
 * - 19: " Frame is too large";
 * - 1A: " Unit powered up in authentication state but POS must resend the InitSecureComm command";
 * - 1B: " The EEPROM may not be initialized because SecCommInterface does not make sense";
 * - 1C: " Problem encoding APDU Module-Specific Status Codes ";
 * - 20: " Unsupported Index (ILM) SAM Transceiver error - problem communicating with the SAM (Key Mgr)";
 * - 21: " Unexpected Sequence Counter in multiple frames for single bitmap (ILM)Length error in data returned from the SAM (Key Mgr)
 * - 22: " Improper bit map (ILM)";
 * - 23: " Request Online Authorization";
 * - 24: " ViVOCard3 raw data read successful";
 * - 25: " Message index not available (ILM) ViVOcomm activate transaction card type (ViVOcomm)";
 * - 26: " Version Information Mismatch (ILM)";
 * - 27: " Not sending commands in correct index message index (ILM)";
 * - 28: " Time out or next expected message not received (ILM)";
 * - 29: " ILM languages not available for viewing (ILM)";
 * - 2A: " Other language not supported (ILM)";
 * - 41: " from 41 to 4F, Module-specific errors for Key Manager";
 *
 * - 50: " Auto-Switch OK";
 * - 51: " Auto-Switch failed";
 * - 70: " Antenna Error 80h Use another card";
 * - 81: " Insert or swipe card";
 * - 90: " Data encryption Key does not exist";
 * - 91: " Data encryption Key KSN exhausted";
 */
void
device_getIDGStatusCodeString(IN int returnCode, OUT char * despcrition);

/**
 * List Directory
 * This command retrieves a directory listing of user accessible files from the reader.
 * @param directoryName Directory Name.  If null, root directory is listed
 * @param directoryNameLen Directory Name Length.  If null, root directory is listed
 * @param recursive Included sub-directories
 * @param onSD 0: use internal memory, 1: use SD card
 * @directory The returned directory information
 * @directoryLen The returned directory information length
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_listDirectory(IN char * directoryName, IN int directoryNameLen, IN int recursive, IN int onSD,
  OUT char * directory, IN_OUT int * directoryLen);

/**
 * Delete Directory
 *  This command deletes an empty directory.  For NEO 2 devices, it will delete the directory even the directory is not empty.
 * @param dirName Complete path of the directory you want to delete.
 *   You do not need to specify the root directory. Indicate subdirectories with a forward slash (/).
 *   For NEO 2 devices, to delete the root directory, simply pass "" with 0 for dirNameLen.
 * @param dirNameLen Directory Name Length.
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_deleteDirectory(IN char * dirName,
  IN int                         dirNameLen);

/**
 * Get Device Memory Usage Information
 * @param freeHeapSize
 *  Free Heap Size: Available heap size
 *
 * @param notFreedBlockCnt
 *  Memory Not Freed Block Count: Memory in use block count
 *
 * @param minEverFreeHeapSize
 *  Minimum Ever Free Heap Size: The lowest ever available heap size
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_getDeviceMemoryUsageInfo(OUT int * freeHeapSize, OUT int * notFreedBlockCnt, OUT int * minEverFreeHeapSize);

/**
 * Get DRS Status
 *
 * Gets the status of DRS(Destructive Reset).
 *
 * @param codeDRS
 *  the data format is [DRS SourceBlk Number] [SourceBlk1] ... [SourceBlkN]
 *  [DRS SourceBlk Number] is 2 bytes, format is NumL NumH. It is Number of [SourceBlkX]
 *  [SourceBlkX] is n bytes, Format is [SourceID] [SourceLen] [SourceData]
 *  [SourceID] is 1 byte
 *  [SourceLen] is 1 byte, it is length of [SourceData]
 *
 * [SourceID]    [SourceLen]  [SourceData]
 *    00                1     01 - Application Error
 *    01                1     01 - Application Error
 *    02                1     0x01 - EMV L2 Configuration Check Value Error
 *                            0x02 - Future Key Check Value Error
 *    10                1     01    - Battery Error
 *    11                1     Bit 0 - Tamper Switch 1 (0-No, 1-Error)
 *                            Bit 1 - Tamper Switch 2 (0-No, 1-Error)
 *                            Bit 2 - Tamper Switch 3 (0-No, 1-Error)
 *                            Bit 3 - Tamper Switch 4 (0-No, 1-Error)
 *                            Bit 4 - Tamper Switch 5 (0-No, 1-Error)
 *                            Bit 5 - Tamper Switch 6 (0-No, 1-Error)
 *
 *    12                1     01  - TemperatureHigh or Low
 *    13                1     01  - Voltage High or Low
 *    1F                4     Reg31~24bits, Reg23~16bits, Reg15~8bits, Reg7~0bits
 *
 *    @param codeDRSLen
 *        the length of codeDRS
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_getDRS(BYTE * codeDRS,
  int *              codeDRSLen);

/**
 * Set the Core Dump Log File Name
 *
 * Set the file name of the log file when the core dump occurs for the firmware
 *
 *
 * @param filename the file name including the path
 * @param filenameLen the length of filename; the maximum length is 99 bytes not including the null character
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_setCoreDumpLogFile(IN char * filename, IN int filenameLen);

/**
 * Output Log
 *
 * Save the log to a file
 *
 *
 * @param filename the file name including the path
 * @param filenameLen the length of filename
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_outputLog(IN char * filename, IN int filenameLen);

/**
 * Get Tamper Status
 *  This command check the tamper status for NEO 2 readers.
 * @param isTampered 1: the reader is tampered. 0: the reader is not tampered.
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_getTamperStatus(OUT int * isTampered);

/**
 * FeliCa Authentication
 *  Provides a key to be used in a follow up FeliCa Read with MAC (3 blocks max) or Write with MAC (1 block max).
 *		This command must be executed before each Read w/MAC or Write w/MAC command
 *
 * @param key 16-byte key used for MAC generation of Read or Write with MAC
 * @param keyLen length of key, must be 16 bytes
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
felica_authentication(IN BYTE * key, IN int keyLen);

/**
 * FeliCa Read with MAC Generation
 *
 *  Reads up to 3 blocks with MAC Generation.  FeliCa Authentication must be performed first
 *
 * @param blockCnt Number of blocks in blockList.  Maximum 3 block requests
 * @param blockList Block to read. Each block in blockList = 2 or 3 bytes of data.
 * @param blockListLen Length of blockList.
 *
 * @param blockData Blocks read will be stored in blockData.  Each block is 16 bytes.
 * @param blockDataLen Length of blockData.
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
felica_readWithMac(IN int blockCnt, IN BYTE * blockList, IN int blockListLen, OUT BYTE * blockData,
  OUT int * blockDataLen);

/**
 * FeliCa Write with MAC Generation
 *
 * Writes a block with MAC Generation.  FeliCa Authentication must be performed first
 *
 * @param blockNum Number of block
 * @param blockData  Block to write.
 * @param blockDataLen  Length of blockData.  Must be 16 bytes.
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
felica_writeWithMac(IN BYTE blockNum, IN BYTE * blockData, IN int blockDataLen);

/**
 * FeliCa Read
 *
 * Reads up to 4 blocks.
 *
 * @param serviceCodeList Service Code List. Each service code in Service Code List = 2 bytes of data
 * @param serviceCodeListLen Length of serviceCodeList
 *
 * @param blockCnt Number of blocks in blockList.  Maximum 4 block requests
 * @param blockList Block to read. Each block in blockList = 2 or 3 bytes of data.
 * @param blockListLen Length of blockList.
 *
 * @param blockData Blocks read will be stored in blockData.  Each block 16 bytes.
 * @param blockDataLen Length of blockData.
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
felica_read(IN BYTE * serviceCodeList, IN int serviceCodeListLen, IN int blockCnt, IN BYTE * blockList,
  IN int blockListLen, OUT BYTE * blockData, OUT int * blockDataLen);

/**
 * FeliCa Write
 *
 * Writes a block
 *
 * @param serviceCodeList Service Code List. Each service code in Service Code List = 2 bytes of data
 * @param serviceCodeListLen Length of serviceCodeList
 * @param blockCnt Number of blocks in blockList.  Currently only support 1 block.
 * @param blockList Block list.  Each block in blockList = 2 or 3 bytes of data.
 * @param blockListLen Length of blockData.
 *
 * @param blockData  Block to write.
 * @param blockDataLen Length of blockData.  Must be 16 bytes.
 *
 * @param respData If successful, the Status Flag (2 bytes) is stored in respData.resData.
 *      Status flag response as explained in FeliCA Lite-S User's Manual, Section 4.5
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
felica_write(IN BYTE * serviceCodeList, IN int serviceCodeListLen, IN int blockCnt, IN BYTE * blockList,
  IN int blockListLen, IN BYTE * blockData, IN int blockDataLen, OUT BYTE * statusFlag,
  OUT int * statusFlagLen);


/**
 * FeliCa Poll for Card
 *
 * Polls for a Felica Card
 *
 * @param systemCode System Code.
 * @param systemCodeLen Length of systemCode.  Must be 2 bytes
 *
 * @param respData response data will be stored in respData.  Poll response as explained in FeliCA Lite-S User's Manual
 * @param respDataLen Length of systemCode.
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
felica_poll(IN BYTE * systemCode, IN int systemCodeLen, OUT BYTE * respData, OUT int * respDataLen);

/**
 * FeliCa Send Command
 *
 * Send a Felica Command
 *
 * @param command Command data from settlement center to be sent to felica card
 * @param commandLen: Length of command data
 *
 * @param respData Response data from felica card.
 * @param respDataLen Length of respData.
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
felica_SendCommand(IN BYTE * command, IN int commandLen, OUT BYTE * respData, OUT int * respDataLen);

/**
 * FeliCa Request Service
 *
 * Request Service for a Felica Card
 *
 * @param nodeCode Node Code List.  Each node 2 bytes
 * @param nodeCodeLen Length of nodeCode.
 * @param respData response data will be stored in respData.  Response as explained in FeliCA Lite-S User's Manual
 * @param respDataLen Length of respData.
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
felica_requestService(IN BYTE * nodeCode, IN int nodeCodeLen, OUT BYTE * respData, OUT int * respDataLen);

/**
 * FeliCa Get Code
 *
 * Start the FeliCa get code process
 * Since the firmware timeout is 180 seconds, the SDK timeout of the command is set to 181 seconds.
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
felica_getCode();

/**
 * FeliCa Cancel Code Entry
 *
 * Cancels FeliCa Get Code request
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
felica_cancelCodeEntry();

// long long timeInMilliseconds(void);
// int felica_SendCommandTest(IN BYTE *command, IN int commandLen, OUT BYTE *respData, OUT int *respDataLen);

/**
 * DEPRECATED : please use config_getSerialNumber_Len(OUT char* sNumber, IN_OUT int *sNumberLen)
 *
 * Polls device for Serial Number
 *
 * @param sNumber  Returns Serial Number; needs to have at least 64 bytes of memory
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString
 *
 */
int
config_getSerialNumber(OUT char * sNumber);

/**
 * Polls device for Serial Number
 *
 * @param sNumber  Returns Serial Number
 * @param sNumberLen  Length of Serial Number
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString
 *
 */
int
config_getSerialNumber_Len(OUT char * sNumber, IN_OUT int * sNumberLen);

/**
 * DEPRECATED : please use config_getModelNumber_Len(OUT char* sNumber, IN_OUT int *sNumberLen)
 *
 * Polls device for Model Number
 *
 * @param sNumber  Returns Model Number; needs to have at least 64 bytes of memory
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString
 *
 */
int
config_getModelNumber(OUT char * sNumber);

/**
 * Polls device for Model Number
 *
 * @param sNumber  Returns Model Number
 * @param sNumber  length of Model Number
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString
 *
 */
int
config_getModelNumber_Len(OUT char * sNumber, IN_OUT int * sNumberLen);

/**
 * Set the timeout duration for regular commands
 * The new timeout value will affect all the functions actually send (sync) commands
 * that doesn’t need to wait for a callback function, such as device_getFirmwareVersion(),
 * device_pingDevice(), device_SendDataCommandNEO(), device_enablePassThrough(), device_setBurstMode(),
 * device_setPollMode(), device_updateFirmware() …etc.
 *
 * @param millisecond  timeout value in milliseconds.  Please consult the firmware team for the proper value.
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
config_setCmdTimeOutDuration(IN int millisecond);

/**
 * Start CTLS Transaction Request
 *
 * Authorizes the CTLS transaction for an ICC card
 *
 * The tags will be returned in the callback routine.
 *
 * @param amount Transaction amount value (tag value 9F02)
 * - SEE IMPORTANT NOTE BELOW
 * @param amtOther Other amount value, if any (tag value 9F03)
 * - SEE IMPORTANT NOTE BELOW
 * @param type Transaction type (tag value 9C).
 * @param timeout Timeout value in seconds.
 * @param tags Any other tags to be included in the request.  Passed as
 * TLV stream.  Example, tag 9F02 with amount 0x000000000100 would be
 * 0x9F0206000000000100.  If tags 9F02 (amount),9F03 (other amount), or 9C
 * (transaction type) are included, they will take priority over these
 * values supplied as individual parameters to this method.
 *
 * @param tagsLen The length of tags data buffer.
 *
 * >>>>>IMPORTANT: parameters for amount and amtOther MUST BE PASSED AS
 *    A DOUBLE VALUE WITH DECIMAL POINT.  Example, do not pass 1, but
 *    instead pass 1.0 or 1.00. Otherwise, results will be
 *    unpredictable
 *
 * @return RETURN_CODE: Values can be parsed with
 * device_getIDGStatusCodeString() Note: if auto poll is on, it will
 * returm the error IDG_P2_STATUS_CODE_COMMAND_NOT_ALLOWED
 *
 * NOTE ON APPLEPAY VAS:
 * To enable ApplePay VAS, first a merchant record must be defined in
 * one of the six available index positions (1-6) using
 * device_setMerchantRecord, then container tag FFEE06 must be sent as
 * part of the additional tags parameter of ctls_startTransaction.
 * Tag FFEE06 must contain tag 9F26 and 9F22, and can optionanally
 * contain tags 9F2B and DF01.  Example
 * FFEE06189F220201009F2604000000009F2B050100000000DF010101
 * 9F22 = two bytes = ApplePay Terminal Applicaiton Version Number.
 *        Hard defined as 0100 for now. (required)
 * 9F26 = four bytes = ApplePay Terminal Capabilities Information (required)
 *  - Byte 1 = RFU
 *  - Byte 2 = Terminal Type
 *  - - Bit 8 = VAS Support     (1=on, 0 = off)
 *  - - Bit 7 = Touch ID Required  (1=on, 0 = off)
 *  - - Bit 6 = RFU
 *  - - Bit 5 = RFU
 *  - - Bit 1,2,3,4
 *  - - - 0 = Payment Terminal
 *  - - - 1 = Transit Terminal
 *  - - - 2 = Access Terminal
 *  - - - 3 = Wireless Handoff Terminal
 *  - - - 4 = App Handoff Terminal
 *  - - - 15 = Other Terminal
 *  - Byte 3 = RFU
 *  - Byte 4 = Terminal Mode
 *  - - 0 = ApplePay VAS OR ApplePay
 *  - - 1 = ApplePay VAS AND ApplePay
 *  - - 2 = ApplePay VAS ONLY
 *  - - 3 = ApplePay ONLY
 *  9F2B = 5 bytes = ApplePay VAS Filter.  Each byte filters for that
 *         specific merchant index  (optional)
 *  DF01 = 1 byte = ApplePay VAS Protocol.  (optional)
 *  - - Bit 1 : 1 = URL VAS, 0 = Full VAS
 *  - - Bit 2 : 1 = VAS Beeps, 0 = No VAS Beeps
 *  - - Bit 3 : 1 = Silent Comm Error, 2 = EMEA Comm Error
 *  - - Bit 4-8 : RFU
 *
 */
int
ctls_startTransaction(IN double amount,
  IN double                     amtOther,
  IN int                        type,
  IN const int                  _timeout,
  IN BYTE *                     tags,
  IN int                        tagsLen);

/**
 * Start CTLS Transaction Request
 *
 * Authorizes the CTLS transaction for an ICC card
 *
 * The tags will be returned in the callback routine.
 *
 * @param timeout Timeout value in seconds.
 * @param tags The tags to be included in the request.  Passed as TLV stream.
 *   Example, tag 9F02 with amount 0x000000000100 would be 0x9F0206000000000100
 *
 * @param tagsLen The length of tags data buffer.
 *
 * >>>>>IMPORTANT: parameters for amount and amtOther MUST BE PASSED AS
 *    A DOUBLE VALUE WITH DECIMAL POINT.  Example, do not pass 1, but
 *    instead pass 1.0 or 1.00. Otherwise, results will be
 *    unpredictable
 *
 * @return RETURN_CODE: Values can be parsed with
 * device_getIDGStatusCodeString() Note: if auto poll is on, it will
 * returm the error IDG_P2_STATUS_CODE_COMMAND_NOT_ALLOWED
 *
 * NOTE ON APPLEPAY VAS:
 * To enable ApplePay VAS, first a merchant record must be defined in
 * one of the six available index positions (1-6) using
 * device_setMerchantRecord, then container tag FFEE06 must be sent as
 * part of the additional tags parameter of ctls_startTransaction.
 * Tag FFEE06 must contain tag 9F26 and 9F22, and can optionanally
 * contain tags 9F2B and DF01.  Example
 * FFEE06189F220201009F2604000000009F2B050100000000DF010101
 * 9F22 = two bytes = ApplePay Terminal Applicaiton Version Number.
 *        Hard defined as 0100 for now. (required)
 * 9F26 = four bytes = ApplePay Terminal Capabilities Information (required)
 *  - Byte 1 = RFU
 *  - Byte 2 = Terminal Type
 *  - - Bit 8 = VAS Support (1=on, 0 = off)
 *  - - Bit 7 = Touch ID Required  (1=on, 0 = off)
 *  - - Bit 6 = RFU
 *  - - Bit 5 = RFU
 *  - - Bit 1,2,3,4
 *  - - - 0 = Payment Terminal
 *  - - - 1 = Transit Terminal
 *  - - - 2 = Access Terminal
 *  - - - 3 = Wireless Handoff Terminal
 *  - - - 4 = App Handoff Terminal
 *  - - - 15 = Other Terminal
 *  - Byte 3 = RFU
 *  - Byte 4 = Terminal Mode
 *  - - 0 = ApplePay VAS OR ApplePay
 *  - - 1 = ApplePay VAS AND ApplePay
 *  - - 2 = ApplePay VAS ONLY
 *  - - 3 = ApplePay ONLY
 *  9F2B = 5 bytes = ApplePay VAS Filter.  Each byte filters for
 *         that specific merchant index  (optional)
 *  DF01 = 1 byte = ApplePay VAS Protocol.  (optional)
 *  - - Bit 1 : 1 = URL VAS, 0 = Full VAS
 *  - - Bit 2 : 1 = VAS Beeps, 0 = No VAS Beeps
 *  - - Bit 3 : 1 = Silent Comm Error, 2 = EMEA Comm Error
 *  - - Bit 4-8 : RFU
 *
 */
int
ctls_activateTransaction(IN const int _timeout,
  IN BYTE *                           tags,
  IN int                              tagsLen);

/**
 * Cancel EMV Transaction
 *
 * Cancels the currently executing EMV transaction.
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
ctls_cancelTransaction();

/**
 * Retrieve Application Data by AID
 *
 * Retrieves the Application Data as specified by the AID name passed as
 * a parameter.
 *
 * @param AID Name of ApplicationID. Must be between 5 and 16 bytes
 * @param AIDLen the length of AID data buffer.
 * @param tlv  The TLV elements of the requested AID
 * @param tlvLen the length of tlv data buffer.
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
ctls_retrieveApplicationData(IN BYTE * AID,
  IN int                               AIDLen,
  OUT BYTE *                           tlv,
  IN_OUT int *                         tlvLen);

/**
 * Set Application Data by AID
 *
 * Sets the Application Data for CTLS as specified by TLV data
 *
 * @param tlv  Application data in TLV format
 *      The first tag of the TLV data must be the group number (FFE4).
 *      The second tag of the TLV data must be the AID (9F06)
 *
 *      Example valid TLV, for Group #2, AID a0000000035010:
 *      "ffe401029f0607a0000000051010ffe10101ffe50110ffe30114ffe20106"
 * @param tlvLen the length of tlv data buffer
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
ctls_setApplicationData(IN BYTE * tlv,
  IN int                          tlvLen);

/**
 * Remove Application Data by AID
 * Removes the Application Data for CTLS as specified by the AID name
 * passed as a parameter
 *
 * @param AID Name of ApplicationID Must be between 5 and 16 bytes
 *
 * @param AIDLen the length of AID data buffer
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
ctls_removeApplicationData(IN BYTE * AID,
  IN int                             AIDLen);

/**
 * Remove All Application Data
 *
 * Removes all the Application Data
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 **/
int
ctls_removeAllApplicationData();

/**
 * Retrieve AID list
 *
 * Returns all the AID names installed on the terminal for CTLS. .
 *
 * @param AIDList  array of AID name byte arrays
 * @param AIDListLen  the length of AIDList array buffer
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
ctls_retrieveAIDList(OUT BYTE * AIDList,
  IN_OUT int *                  AIDListLen);

/**
 * Retrieve Terminal Data
 *
 *
 * Retrieves the Terminal Data for CTLS. This is configuration group 0
 * (Tag FFEE - > FFEE0100).  The terminal data can also be retrieved
 * by ctls_getConfigurationGroup(0).
 *
 * @param tlv Response returned as a TLV
 * @param tlvLen the length of tlv data buffer
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
ctls_retrieveTerminalData(OUT BYTE * tlv,
  IN_OUT int *                       tlvLen);

/**
 * Set Terminal Data
 *
 * Sets the Terminal Data for CTLS as specified by the TLV.  The first
 *      TLV must be Configuration Group Number (Tag FFE4).  The
 *      terminal global data is group 0, so the first TLV would be
 *      FFE40100.  Other groups can be defined using this method (1 or
 *      greater), and those can be retrieved with
 *      ctls_getConfigurationGroup(int group), and deleted with
 *      ctls_removeConfigurationGroup(int group).  You cannot delete
 *      group 0.
 *
 * @param tlv TerminalData configuration file
 * @param tlvLen the length of tlv data buffer
 * @retval RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
ctls_setTerminalData(IN BYTE * tlv,
  IN int                       tlvLen);

/**
 * Retrieve Certificate Authority Public Key
 *
 *
 * Retrieves the CAPK for CTLS as specified by the RID/Index passed as a
 * parameter.
 *
 * @param capk 6 bytes CAPK = 5 bytes RID + 1 byte Index
 * @param capkLen the length of capk data buffer
 * @param key Response returned as a CAKey format:
 *  [5 bytes RID][1 byte Index][1 byte Hash Algorithm][1 byte Encryption Algorithm]
 *  [20 bytes HashValue][4 bytes Public Key Exponent][2 bytes Modulus Length][Variable bytes Modulus]
 *  Where:
 *   - Hash Algorithm: The only algorithm supported is SHA-1.The value is set to 0x01
 *   - Encryption Algorithm: The encryption algorithm in which this key is used. Currently support only one type: RSA. The value is set to 0x01.
 *   - HashValue: Which is calculated using SHA-1 over the following fields: RID & Index & Modulus & Exponent
 *   - Public Key Exponent: Actually, the real length of the exponent is either one byte or 3 bytes. It can have two values: 3 (Format is 0x00 00 00 03), or  65537 (Format is 0x00 01 00 01)
 *   - Modulus Length: LenL LenH Indicated the length of the next field.
 *   - Modulus: This is the modulus field of the public key. Its length is specified in the field above.
 * @param keyLen the length of key data buffer
 * * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
ctls_retrieveCAPK(IN BYTE * capk,
  IN int                    capkLen,
  OUT BYTE *                key,
  IN_OUT int *              keyLen);

/**
 * Set Certificate Authority Public Key
 *
 * Sets the CAPK as specified by the CAKey structure
 *
 * @param capk CAKey format:
 *  [5 bytes RID][1 byte Index][1 byte Hash Algorithm][1 byte Encryption Algorithm][20 bytes HashValue][4 bytes Public Key Exponent][2 bytes Modulus Length][Variable bytes Modulus]
 *  Where:
 *   - Hash Algorithm: The only algorithm supported is SHA-1.The value is set to 0x01
 *   - Encryption Algorithm: The encryption algorithm in which this key is used. Currently support only one type: RSA. The value is set to 0x01.
 *   - HashValue: Which is calculated using SHA-1 over the following fields: RID & Index & Modulus & Exponent
 *   - Public Key Exponent: Actually, the real length of the exponent is either one byte or 3 bytes. It can have two values: 3 (Format is 0x00 00 00 03), or  65537 (Format is 0x00 01 00 01)
 *   - Modulus Length: LenL LenH Indicated the length of the next field.
 *   - Modulus: This is the modulus field of the public key. Its length is specified in the field above.
 * @param capkLen the length of capk data buffer
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
ctls_setCAPK(IN BYTE * capk,
  IN int               capkLen);

/**
 * Remove Certificate Authority Public Key
 *
 * Removes the CAPK as specified by the RID/Index
 *
 * @param capk 6 byte CAPK =  5 bytes RID + 1 byte INDEX
 * @param capkLen the length of capk data buffer
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
ctls_removeCAPK(IN BYTE * capk,
  IN int                  capkLen);

/**
 * Remove All Certificate Authority Public Key
 *
 * Removes all the CAPK
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
ctls_removeAllCAPK();

/**
 * Retrieve the Certificate Authority Public Key list
 *
 * Returns all the CAPK RID and Index installed on the terminal.
 *
 * @param keys [key1][key2]...[keyn], each key 6 bytes where
 *  key = 5 bytes RID + 1 byte index
 * @param keysLen the length of keys data buffer
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
ctls_retrieveCAPKList(OUT BYTE * keys,
  IN_OUT int *                   keysLen);

/**
 * Set Configuration Group
 *
 * Sets the Configuration Group for CTLS as specified by the TLV data
 *
 * @param tlv  Configuration Group Data in TLV format
 *        The first tag of the TLV data must be the group number (DFEE2D).
 *        A second tag must exist
 * @param tlvLen the length of tlv data buffer
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
ctls_setConfigurationGroup(IN BYTE * tlv,
  IN int                             tlvLen);

/**
 * Get Configuration Group
 *
 * Retrieves the Configuration for the specified Group.
 *
 * @param group Configuration Group
 * @param tlv return data
 * @param tlvLen the length of tlv data buffer
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
ctls_getConfigurationGroup(IN int group,
  OUT BYTE *                      tlv,
  OUT int *                       tlvLen);

/**
 * Retrieve All Configuration Groups
 *
 * Returns all the Configuration Groups installed on the terminal for CTLS
 *
 * @param tlv  The TLV elements data
 * @param tlvLen the length of tlv data buffer.
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
ctls_getAllConfigurationGroups(OUT BYTE * tlv,
  IN_OUT int *                            tlvLen);

/**
 * Remove Configuration Group
 *
 * Removes the Configuration as specified by the Group.  Must not by group 0
 *
 * @param group Configuration Group
 * @retval RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
ctls_removeConfigurationGroup(int group);

/**
 * DEPRECATED : please use emv_getEMVKernelVersion_Len(OUT char* version, IN_OUT int *versionLen)
 *
 * Polls device for EMV Kernel Version
 *
 * @param version Response returned of Kernel Version; needs to have at least 128 bytes of memory.
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
emv_getEMVKernelVersion(OUT char * version);

/**
 * Polls device for EMV Kernel Version
 *
 * @param version Response returned of Kernel Version
 *
 * @param versionLen Length of version
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
emv_getEMVKernelVersion_Len(OUT char * version, IN_OUT int * versionLen);

/**
 * Get EMV Kernel check value info
 *
 * @param checkValue Response returned of Kernel check value info
 * @param checkValueLen the length of checkValue
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString
 */
int
emv_getEMVKernelCheckValue(OUT BYTE * checkValue,
  IN_OUT int *                        checkValueLen);

/**
 * Get EMV Kernel configuration check value info
 *
 * @param checkValue Response returned of Kernel configuration check value info
 * @param checkValueLen the length of checkValue
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
emv_getEMVConfigurationCheckValue(OUT BYTE * checkValue,
  IN_OUT int *                               checkValueLen);

/**
 * Allow fallback for EMV transactions.  Default is TRUE
 * @param allow TRUE = allow fallback, FALSE = don't allow fallback
 */
void
emv_allowFallback(IN int allow);

/**
 * Enables authenticate for EMV transactions.  If a emv_startTranaction
 *        results in code 0x0010 (start transaction success), then
 *        emv_authenticateTransaction can automatically execute if
 *        parameter is set to TRUE
 * @param authenticate TRUE = auto authenticate, FALSE = manually authenticate
 */
void
emv_setAutoAuthenticateTransaction(IN int authenticate);

/**
 * Enables complete for EMV transactions.  If a
 * emv_authenticateTranaction results in code 0x0004 (go online), then
 * emv_completeTransaction can automatically execute if parameter is
 * set to TRUE
 * @param complete TRUE = auto complete, FALSE = manually complete
 */
void
emv_setAutoCompleteTransaction(IN int complete);

/**
 * Gets auto authenticate value for EMV transactions.
 * @return RETURN_CODE:  TRUE = auto authenticate, FALSE = manually authenticate
 */
int
emv_getAutoAuthenticateTransaction();

/**
 * Gets auto complete value for EMV transactions.
 * @return RETURN_CODE:  TRUE = auto complete, FALSE = manually complete
 */
int
emv_getAutoCompleteTransaction();

/**
 * Set EMV Transaction Parameters
 *
 * Set the parameters to be used on EMV transactions for an ICC card
 * when Auto Poll is on
 *
 * The tags will be returned in the callback routine.
 *
 * @param amount Transaction amount value  (tag value 9F02)
 * @param amtOther Other amount value, if any  (tag value 9F03)
 * @param type Transaction type (tag value 9C).
 * @param timeout Timeout value in seconds.
 * @param tags Any other tags to be included in the request (Maximum
 *   Length = 500 bytes).  Passed as a string.  Example, tag 9F02 with
 *   amount 0x000000000100 would be "9F0206000000000100" If tags 9F02
 *   (amount),9F03 (other amount), or 9C (transaction type) are
 *   included, they will take priority over these values supplied as
 *   individual parameters to this method.  Note: To request tags to
 *   be included in default response, use tag DFEE1A, and specify tag
 *   list.  Example four tags 9F02, 9F36, 95, 9F37 to be included in
 *   response = DFEE1A079F029F36959F37
 * @param tagsLen the length of tags
 */
void
emv_setTransactionParameters(IN double amount,
  IN double                            amtOther,
  IN int                               type,
  IN int                               timeout,
  IN BYTE *                            tags,
  IN int                               tagsLen);

/**
 * Start EMV Transaction Request
 *
 * Authorizes the EMV transaction for an ICC card
 *
 * The tags will be returned in the callback routine.
 *
 * @param amount Transaction amount value (tag value 9F02) - SEE
 * IMPORTANT NOTE BELOW
 *
 * @param amtOther Other amount value, if any (tag value 9F03) - SEE
 * IMPORTANT NOTE BELOW
 * @param exponent Number of characters after decimal point
 * @param type Transaction type (tag value 9C).
 * @param timeout Timeout value in seconds.
 * @param tags Any other tags to be included in the request.  Passed
 * as a TLV stream.  Example, tag 9F02 with amount 0x000000000100 would
 * be 0x9F0206000000000100 If tags 9F02 (amount),9F03 (other amount), or
 * 9C (transaction type) are included, they will take priority over
 * these values supplied as individual parameters to this method.
 * @param tagsLen Length of tags
 * @param forceOnline TRUE = do not allow offline approval,  FALSE = allow
 * ICC to approve offline if terminal capable
 *
 * Note: To request tags to be included in default response, use tag
 * DFEE1A, and specify tag list.  Example four tags 9F02, 9F36, 95, 9F37
 * to be included in response = DFEE1A079F029F36959F37
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString
 *>>>>>IMPORTANT: parameters for amount and amtOther MUST BE PASSED AS
 *     A DOUBLE VALUE WITH DECIMAL POINT.  Example, do not pass 1, but
 *     instead pass 1.0 or 1.00. Otherwise, results will be
 *     unpredictable
 */
int
emv_startTransaction(IN double amount,
  IN double                    amtOther,
  IN int                       exponent,
  IN int                       type,
  IN int                       timeout,
  IN BYTE *                    tags,
  IN int                       tagsLen,
  IN int                       forceOnline);

/**
 * Activate EMV Transaction Request
 *
 * Authorizes the EMV transaction for an ICC card
 *
 * The tags will be returned in the callback routine.
 *
 * @param timeout Timeout value in seconds.
 * @param tags Tags to be included in the request.  Passed as a TLV stream.
 * Example, tag 9F02 with amount 0x000000000100 would be 0x9F0206000000000100
 * @param tagsLen Length of tags
 * @param forceOnline TRUE = do not allow offline approval, FALSE =
 * allow ICC to approve offline if terminal capable
 * Note: To request tags to be included in default response, use tag
 * DFEE1A, and specify tag list.  Example four tags 9F02, 9F36, 95,
 * 9F37 to be included in response = DFEE1A079F029F36959F37
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_activateTransaction(IN int timeout,
  IN BYTE *                    tags,
  IN int                       tagsLen,
  IN int                       forceOnline);

/**
 * Authenticate EMV Transaction Request
 *
 * Authenticates the EMV transaction for an ICC card.  Execute this after
 * receiving response with result code 0x10 to emv_startTransaction
 *
 * The tags will be returned in the callback routine.
 *
 * @param updatedTLV  TLV stream that can be used to update the following values:
 *   - 9F02: Amount
 *   - 9F03: Other amount
 *   - 9C: Transaction type
 *   - 5F57: Account type
 *
 *  In addition tag DFEE1A can be sent to specify tag list to include
 *  in results. Example four tags 9F02, 9F36, 95, 9F37 to be included
 *  in response = DFEE1A079F029F36959F37
 * @param updatedTLVLen
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_authenticateTransaction(IN BYTE * updatedTLV,
  IN int                              updatedTLVLen);

/**
 * Authenticate EMV Transaction Request with Timeout
 *
 * Authenticates the EMV transaction for an ICC card.  Execute this after receiving response with result code 0x10 to emv_startTransaction
 *
 * The tags will be returned in the callback routine.
 *
 * @param timeout Timeout value in seconds.
 * @param updatedTLV  TLV stream that can be used to update the following values:
 *   - 9F02: Amount
 *   - 9F03: Other amount
 *   - 9C: Transaction type
 *   - 5F57: Account type
 *  In addition tag DFEE1A can be sent to specify tag list to include in results. Example four tags 9F02, 9F36, 95, 9F37 to be included in response = DFEE1A079F029F36959F37
 * @param updatedTLVLen
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_authenticateTransactionWithTimeout(IN int timeout, IN BYTE * updatedTLV, IN int updatedTLVLen);

/**
 * Complete EMV Transaction Request
 *
 * Completes the EMV transaction for an ICC card when online
 * authorization request is received from emv_authenticateTransaction
 *
 * The tags will be returned in the callback routine.
 *
 * @param commError Communication error with host.  Set to TRUE(1) if
 * host was unreachable, or FALSE(0) if host response received.  If
 * Communication error, authCode, iad, tlvScripts can be null.
 * @param authCode Authorization code from host.  Two bytes.  Example
 * 0x3030.  (Tag value 8A).  Required
 * @param authCodeLen the length of authCode
 * @param iad Issuer Authentication Data, if any.  Example
 * 0x11223344556677883030 (tag value 91).
 * @param iadLen the length of iadLen
 * @param tlvScripts 71/72 scripts, if any
 * @param tlvScriptsLen the length of tlvScriptsLen
 * @param tlv  Additional TLV data to return with transaction results (if any)
 * @param tlvLen the length of tlv
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_completeTransaction(IN int commError,
  IN BYTE *                    authCode,
  IN int                       authCodeLen,
  IN BYTE *                    iad,
  IN int                       iadLen,
  IN BYTE *                    tlvScripts,
  IN int                       tlvScriptsLen,
  IN BYTE *                    tlv,
  IN int                       tlvLen);

/**
 * Cancel EMV Transaction
 *
 * Cancels the currently executing EMV transaction.
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_cancelTransaction();

/**
 * Retrieve Transaction Results
 *
 * Retrieves specified EMV tags from the currently executing transaction.
 *
 * @param tags Tags to be retrieved.  Example 0x9F028A will retrieve
 * tags 9F02 and 8A
 * @param tagsLen Length of tag list
 * @param cardData All requested tags returned as unencrypted, encrypted
 * and masked TLV data in IDTTransactionData object
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_retrieveTransactionResult(IN BYTE * tags,
  IN int                                tagsLen,
  OUT IDTTransactionData *              cardData);

/**
 * Retrieve Application Data by AID
 *
 * Retrieves the Application Data as specified by the AID name passed as
 * a parameter.
 *
 * @param AID Name of ApplicationID. Must be between 5 and 16 bytes
 * @param AIDLen the length of AID data buffer.
 * @param tlv  The TLV elements of the requested AID
 * @param tlvLen the length of tlv data buffer.
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_retrieveApplicationData(IN BYTE * AID,
  IN int                              AIDLen,
  OUT BYTE *                          tlv,
  IN_OUT int *                        tlvLen);

/**
 * Set Application Data by AID
 *
 * Sets the Application Data as specified by the application name and TLV data
 *
 * @param name Application name, 10-32 ASCII hex characters representing
 * 5-16 bytes  Example "a0000000031010"
 * @param nameLen the length of name data buffer of Application name,
 * @param tlv  Application data in TLV format
 * @param tlvLen the length of tlv data buffer
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_setApplicationData(IN BYTE * name,
  IN int                         nameLen,
  IN BYTE *                      tlv,
  IN int                         tlvLen);

/**
 * Set Application Data by TLV
 *
 * Sets the Application Data as specified by the TLV data
 * @param tlv  Application data in TLV format
 *              The first tag of the TLV data must be the group number (DFEE2D).
 *              The second tag of the TLV data must be the AID (9F06)
 *              Example valid TLV, for Group #2, AID a0000000035010:
 *              "dfee2d01029f0607a0000000051010ffe10101ffe50110ffe30114ffe20106"
 * @param tlvLen the length of tlv data buffer
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_setApplicationDataTLV(IN BYTE * tlv,
  IN int                            tlvLen);

/**
 * Remove Application Data by AID
 * Removes the Application Data for CTLS as specified by the AID name passed
 * as a parameter
 * @param AID Name of ApplicationID Must be between 5 and 16 bytes
 * @param AIDLen the length of AID data buffer
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_removeApplicationData(IN BYTE * AID,
  IN int                            AIDLen);

/**
 * Remove All Application Data
 *
 * Removes all the Application Data
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_removeAllApplicationData();

/**
 * Retrieve AID list
 *
 * Returns all the AID names installed on the terminal for CTLS. .
 *
 * @param AIDList  array of AID name byte arrays
 * @param AIDListLen  the length of AIDList array buffer
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_retrieveAIDList(OUT BYTE * AIDList,
  IN_OUT int *                 AIDListLen);

/**
 * Retrieve Terminal Data
 *
 * Retrieves the Terminal Data for CTLS. This is configuration group 0
 * (Tag FFEE - > FFEE0100).  The terminal data can also be retrieved
 * by ctls_getConfigurationGroup(0).
 * @param tlv Response returned as a TLV
 * @param tlvLen the length of tlv data buffer
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_retrieveTerminalData(OUT BYTE * tlv,
  IN_OUT int *                      tlvLen);

/**
 * Set Terminal Data
 *
 * Sets the Terminal Data for CTLS as specified by the TLV.  The first
 * TLV must be Configuration Group Number (Tag FFE4).  The terminal
 * global data is group 0, so the first TLV would be FFE40100.  Other
 * groups can be defined using this method (1 or greater), and those
 * can be retrieved with emv_getConfigurationGroup(int group), and
 * deleted with emv_removeConfigurationGroup(int group).  You cannot
 * delete group 0.
 *
 * @param tlv TerminalData configuration file
 * @param tlvLen the length of tlv data buffer
 * @retval RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_setTerminalData(IN BYTE * tlv,
  IN int                      tlvLen);

/**
 * Sets the terminal major configuration in ICS .
 * @param configuration A configuration value, range 1-23
 * - 1 = 1C
 * - 2 = 2C
 * - 3 = 3C
 * - 4 = 4C
 * - 5 = 5C
 * ...
 * - 23 = 23C
 *
 *  @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_setTerminalMajorConfiguration(IN int configuration);

/**
 * Retrieve Certificate Authority Public Key
 *
 * Retrieves the CAPK for CTLS as specified by the RID/Index passed as a
 * parameter.
 *
 * @param capk 6 bytes CAPK = 5 bytes RID + 1 byte Index
 * @param capkLen the length of capk data buffer
 * @param key Response returned as a CAKey format:
 *  [5 bytes RID][1 byte Index][1 byte Hash Algorithm][1 byte Encryption Algorithm]
 *  [20 bytes HashValue][4 bytes Public Key Exponent][2 bytes Modulus Length][Variable bytes Modulus]
 *  Where:
 *   - Hash Algorithm: The only algorithm supported is SHA-1.The value is set to 0x01
 *   - Encryption Algorithm: The encryption algorithm in which this key is used. Currently support only one type: RSA. The value is set to 0x01.
 *   - HashValue: Which is calculated using SHA-1 over the following fields: RID & Index & Modulus & Exponent
 *   - Public Key Exponent: Actually, the real length of the exponent is either one byte or 3 bytes. It can have two values: 3 (Format is 0x00 00 00 03), or  65537 (Format is 0x00 01 00 01)
 *   - Modulus Length: LenL LenH Indicated the length of the next field.
 *   - Modulus: This is the modulus field of the public key. Its length is specified in the field above.
 * @param keyLen the length of key data buffer
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_retrieveCAPK(IN BYTE * capk,
  IN int                   capkLen,
  OUT BYTE *               key,
  IN_OUT int *             keyLen);

/**
 * Set Certificate Authority Public Key
 *
 * Sets the CAPK as specified by the CAKey structure
 *
 * @param capk CAKey format:
 *  [5 bytes RID][1 byte Index][1 byte Hash Algorithm][1 byte Encryption Algorithm][20 bytes HashValue][4 bytes Public Key Exponent][2 bytes Modulus Length][Variable bytes Modulus]
 *  Where:
 *   - Hash Algorithm: The only algorithm supported is SHA-1.The value is set to 0x01
 *   - Encryption Algorithm: The encryption algorithm in which this key is used. Currently support only one type: RSA. The value is set to 0x01.
 *   - HashValue: Which is calculated using SHA-1 over the following fields: RID & Index & Modulus & Exponent
 *   - Public Key Exponent: Actually, the real length of the exponent is either one byte or 3 bytes. It can have two values: 3 (Format is 0x00 00 00 03), or  65537 (Format is 0x00 01 00 01)
 *   - Modulus Length: LenL LenH Indicated the length of the next field.
 *   - Modulus: This is the modulus field of the public key. Its length is specified in the field above.
 * @param capkLen the length of capk data buffer
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_setCAPK(IN BYTE * capk,
  IN int              capkLen);

/**
 * Remove Certificate Authority Public Key
 *
 * Removes the CAPK as specified by the RID/Index
 *
 * @param capk 6 byte CAPK =  5 bytes RID + 1 byte INDEX
 * @param capkLen the length of capk data buffer
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_removeCAPK(IN BYTE * capk,
  IN int                 capkLen);

/**
 * Remove All Certificate Authority Public Key
 *
 * Removes all the CAPK
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
emv_removeAllCAPK();

/**
 * Retrieve the Certificate Authority Public Key list
 *
 * Returns all the CAPK RID and Index installed on the terminal.
 *
 * @param keys [key1][key2]...[keyn], each key 6 bytes where
 *  key = 5 bytes RID + 1 byte index
 * @param keysLen the length of keys data buffer
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_retrieveCAPKList(OUT BYTE * keys,
  IN_OUT int *                  keysLen);

/**
 * Retrieve the Certificate Revocation List
 *
 * Returns the CRL entries on the terminal.
 *
 * @param list [CRL1][CRL2]...[CRLn], each CRL 9 bytes where
 *  CRL = 5 bytes RID + 1 byte index + 3 bytes serial number
 * @param lssLen the length of list data buffer
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_retrieveCRL(OUT BYTE * list,
  IN_OUT int *             lssLen);

/**
 * Set Certificate Revocation List
 *
 * Sets the CRL
 *
 * @param list CRL Entries containing the RID, Index, and serial numbers to set
 *  [CRL1][CRL2]...[CRLn] where each [CRL]  is 9 bytes:
 *  [5 bytes RID][1 byte CAPK Index][3 bytes serial number]
 * @param lsLen the length of list data buffer
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_setCRL(IN BYTE * list,
  IN int             lsLen);

/**
 * Retrieve the Certificate Revocation List
 *
 * Returns the CRL entries on the terminal.
 *
 * @param list [CRL1][CRL2]...[CRLn], each CRL 9 bytes where
 *  CRL = 5 bytes RID + 1 byte index + 3 bytes serial number
 * @param lssLen the length of list data buffer
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_removeCRL(IN BYTE * list,
  IN int                lsLen);

/**
 * Remove All Certificate Revocation List Entries
 *
 * Removes all CRLEntry entries
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
emv_removeAllCRL();

/**
 * Get Reader Status
 *
 * Returns the reader status
 *
 * @param status Pointer that will return with the ICCReaderStatus results.
 *    bit 0:  0 = ICC Power Not Ready, 1 = ICC Powered
 *    bit 1:  0 = Card not seated, 1 = card seated
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString
 */
int
icc_getICCReaderStatus(OUT BYTE * status);

/**
 * Power On ICC
 *
 * Power up the currently selected microprocessor card in the ICC reader
 * @param ATR, the ATR data response when succeeded power on ICC,
 * @param inLen, the length of ATR data
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
icc_powerOnICC(OUT BYTE * ATR,
  IN_OUT int *            inLen);

/**
 * Power Off ICC
 *
 * Powers down the ICC
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 * If Success, empty
 * If Failure, ASCII encoded data of error string
 */
int
icc_powerOffICC();

/**
 * Exchange APDU with plain text
 * For Non-SRED Augusta Only
 *
 * Sends an APDU packet to the ICC.  If successful, response is the
 * APDU data in response parameter.
 *
 * @param c_APDU    APDU data packet
 * @param cLen     APDU data packet length
 * @param reData    Unencrypted APDU response
 * @param reLen      Unencrypted APDU response data length
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString
 */
int
icc_exchangeAPDU(IN BYTE * c_APDU,
  IN int                   cLen,
  OUT BYTE *               reData,
  IN_OUT int *             reLen);

/**
 * Create Screen
 *
 *  Creates a new screen.
 *  @param screenName  Screen name to use.
 *  @param screenNameLen  Length of screenName.
 *  @param screenID  Screen ID that was created.
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
lcd_createScreen(IN char * screenName, IN int screenNameLen, OUT int * ScreenID);

/**
 * Destroy Screen
 *
 *  Destroys a previously created inactive screen.  The screen cannot be active
 *    @param screenName  Screen name to destroy.  The screen number is assigned with lcd_createScreen.
 *    @param screenNameLen  Length of screenName.
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
lcd_destroyScreen(IN char * screenName, IN int screenNameLen);

/**
 * Get Active Screen
 *
 *  Returns the active screen ID.
 *    @param screenName  Screen name this is active.
 *    @param screenNameLen  Length of screenName.
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
lcd_getActiveScreen(OUT char * screenName, IN_OUT int * screenNameLen);

/**
 * Show Screen
 *
 *  Displays and makes active a previously created screen.
 *    @param screenName  Screen to display.  The screen name is defined with lcd_createScreen.
 *    @param screenNameLen  Length of screenName.
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
lcd_showScreen(IN char * screenName, IN int screenNameLen);

/**
 * Get Button Event
 *
 *  Reports back the ID of the button that encountered a click event after the last Get Button Event.
 *
 *    @param screenID  Screen ID of the last clicked button
 *    @param objectID  Button ID of the last clicked button
 *    @param screenName  Screen Name of the last clicked button
 *    @param screenNameLen  Length of screenName
 *    @param objectName  Button Name of the last clicked button
 *    @param objectNameLen  Length of objectName
 *    @param isLongPress  1 = Long Press, 0 = Short Press
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
lcd_getButtonEvent(OUT int * screenID, OUT int * objectID, OUT char * screenName, IN_OUT int * screenNameLen,
  OUT char * objectName, IN_OUT int * objectNameLen, OUT int * isLongPress);

/**
 * Add Button
 *
 *  Adds a button to a selected screen.  Must execute lcd_createScreen first to establish a screen to draw on.
 *  The SDK timeout of the command is set to 6 seconds.
 *
 *    @param screenName Screen name that will be the target of add button
 *    @param screenNameLen  Length of screenName
 *    @param buttonName Button name that will be the target of add button
 *    @param buttonNameLen  Length of buttonName
 *    @param type Button Type
 *      - Large = 0x01
 *      - Medium = 0x02
 *      - Invisible = 0x03  (70px by 60 px)
 *    @param alignment Position for Button
 *      - 0 = Display object at the horizon center of specified y, while x ignored
 *      - 1 = Display object at specified x andy
 *      - 2 = Display object at center of screen, x, y are both ignored
 *      - 3 = Display object at left of the screen of specified y, while x ignored
 *      - 4 = Display object at right of the screen of specified y, while x ignored
 *    @param xCord  x-coordinate for Button, range 0-271
 *    @param yCord  y-coordinate for Button, range 0-479
 *    @param label Label to show on the button.  Required for Large/Medium buttons.  Not used for Invisible buttons.
 *    @param labelLen  Length of label
 *    @param returnItem The item includes screen ID, object ID, top-left x-coordinate, top-left y-coordinate, bottom-right x-coordinate, bottom-left y-coordinate, which are all assigned to the created button
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
lcd_addButton(IN char * screenName, IN int screenNameLen, IN char * buttonName, IN int buttonNameLen, IN BYTE type,
  IN BYTE alignment, IN int xCord, IN int yCord, IN char * label, IN int labelLen,
  OUT IDTLCDItem * returnItem);

/**
 * Add Ethernet Settings
 *
 *  Adds an Ethernet settings to a selected screen.  Must execute lcd_createScreen first to establish a screen to draw on.
 *  The SDK timeout of the command is set to 6 seconds.
 *
 *    @param screenName Screen name that will be the target of add widget
 *    @param screenNameLen  Length of screenName
 *    @param objectName Object name that will be the target of add widget
 *    @param objectNameLen  Length of objectName
 *    @param alignment  Position for widget
 *      - 0 = Display object at the horizon center of specified y, while x ignored
 *      - 1 = Display object at specified x and y
 *      - 2 = Display object at center of screen, x, y are both ignored
 *      - 3 = Display object at left of the screen of specified y, while x ignored
 *      - 4 = Display object at right of the screen of specified y, while x ignored
 *    @param xCord  x-coordinate for widget, range 0-271
 *    @param yCord  y-coordinate for widget, range 0-479
 *    @param returnItem The item includes screen ID, object ID, top-left x-coordinate, top-left y-coordinate, bottom-right x-coordinate, bottom-left y-coordinate, which are all assigned to the created widget
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *   Item  | Maximum can be created for each screen
 *   ----- | -----
 *   Text Area | 20
 *   Large Button | 8
 *   Medium Button | 16
 *   Invisible Button | 16
 *   Numeric Entry | 1
 *   Ethernet Setting | 1
 *   Led widget | 1
 *   image | 20
 */
int
lcd_addEthernet(IN char * screenName, IN int screenNameLen, IN char * objectName, IN int objectNameLen,
  IN BYTE alignment, IN int xCord, IN int yCord, OUT IDTLCDItem * returnItem);

/**
 * Add LED
 *
 *  Adds a LED widget to a selected screen.  Must execute lcd_createScreen first to establish a screen to draw on.
 *  The SDK timeout of the command is set to 6 seconds.
 *
 *    @param screenName Screen name that will be the target of add LED
 *    @param screenNameLen  Length of screenName
 *    @param objectName Object name that will be the target of add LED
 *    @param objectNameLen  Length of objectName
 *    @param alignment  Position for LED
 *      - 0 = Display object at the horizon center of specified y, while x ignored
 *      - 1 = Display object at specified x andy
 *      - 2 = Display object at center of screen, x, y are both ignored
 *      - 3 = Display object at left of the screen of specified y, while x ignored
 *      - 4 = Display object at right of the screen of specified y, while x ignored
 *    @param xCord  x-coordinate for LED, range 0-271
 *    @param yCord  y-coordinate for LED, range 0-479
 *    @param returnItem The item includes screen ID, object ID, top-left x-coordinate, top-left y-coordinate, bottom-right x-coordinate, bottom-left y-coordinate, which are all assigned to the created widget
 *    @param LED Must be 4 bytes, LED 0 = byte 0, LED 1 = byte 1, LED 2 = byte 2, LED 3 = byte 3
 *       - Value 0 = LED OFF
 *       - Value 1 = LED Green
 *       - Value 2 = LED Yellow
 *       - Value 3 = LED Red
 *    @param LEDLen  Length of LED
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *  Item | Maximum can be created for each screen
 *  ----- | -----
 *  Text Area | 20
 *  Large Button | 8
 *  Medium Button | 16
 *  Invisible Button | 16
 *  Numeric Entry | 1
 *  Ethernet Setting | 1
 *  Led widget | 1
 *  image | 20
 */
int
lcd_addLED(IN char * screenName, IN int screenNameLen, IN char * objectName, IN int objectNameLen, IN BYTE alignment,
  IN int xCord, IN int yCord, OUT IDTLCDItem * returnItem, IN BYTE * LED, IN int LEDLen);

/**
 * Add text
 *
 *  Adds a text component to a selected screen.  Must execute lcd_createScreen first to establish a screen to draw on.
 *  The SDK timeout of the command is set to 6 seconds.
 *
 *    @param screenName Screen name that will be the target of add text
 *    @param screenNameLen  Length of screenName
 *    @param objectName Object name that will be the target of add text
 *    @param objectNameLen  Length of objectName
 *    @param alignment  Position for Text
 *      - 0 = Display object at the horizon center of specified y, while x ignored
 *      - 1 = Display object at specified x andy
 *      - 2 = Display object at center of screen, x, y are both ignored
 *      - 3 = Display object at left of the screen of specified y, while x ignored
 *      - 4 = Display object at right of the screen of specified y, while x ignored
 *    @param xCord  x-coordinate for Text, range 0-271
 *    @param yCord  y-coordinate for Text, range 0-479
 *    @param width  Width of text area
 *    @param height  Height of text area
 *    @param fontID Font ID
 *    @param color Four bytes for color, example, Blue = 0xFF000000, Black = 0x00000000
 *      - Byte 0 = B
 *      - Byte 1 = G
 *      - Byte 2 = R
 *      - Byte 3 = Reserved
 *    @param colorLen  Length of color
 *    @param label Label to show on the text
 *    @param labelLen  Length of label
 *    @param returnItem The item includes screen ID, object ID, top-left x-coordinate, top-left y-coordinate, bottom-right x-coordinate, bottom-left y-coordinate, which are all assigned to the created text area
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 * Font ID | Typography Name | Font | Size
 * ----- | ----- | ----- | -----
 * 0 | RoundBold_12 | RoundBold.ttf | 12
 * 1 | RoundBold_18 | RoundBold.ttf | 18
 * 2 | RoundBold_24 | RoundBold.ttf | 24
 * 3 | RoundBold_36 | RoundBold.ttf | 36
 * 4 | RoundBold_48 | RoundBold.ttf | 48
 * 5 | RoundBold_60 | RoundBold.ttf | 60
 * 6 | RoundBold_72 | RoundBold.ttf | 72
 * 7 | RoundCondensedBold_12 | RoundCondensedBold.ttf | 12
 * 8 | RoundCondensedBold_18 | RoundCondensedBold.ttf | 18
 * 9 | RoundCondensedBold_24 | RoundCondensedBold.ttf | 24
 * 10 | RoundCondensedBold_36 | RoundCondensedBold.ttf | 36
 * 11 | RoundCondensedBold_48 | RoundCondensedBold.ttf | 48
 * 12 | RoundCondensedBold_60 | RoundCondensedBold.ttf | 60
 * 13 | RoundCondensedBold_72 | RoundCondensedBold.ttf | 72
 * 14 | RoundCondensedMedium_12 | RoundCondensedMedium_0.ttf | 12
 * 15 | RoundCondensedMedium_18 | RoundCondensedMedium_0.ttf | 18
 * 16 | RoundCondensedMedium_24 | RoundCondensedMedium_0.ttf | 24
 * 17 | RoundCondensedMedium_36 | RoundCondensedMedium_0.ttf | 36
 * 18 | RoundCondensedMedium_48 | RoundCondensedMedium_0.ttf | 48
 * 19 | RoundCondensedMedium_60 | RoundCondensedMedium_0.ttf | 60
 * 20 | RoundCondensedMedium_72 | RoundCondensedMedium_0.ttf | 72
 * 21 | RoundCondensedSemibold_12 | RoundCondensedSemibold.ttf | 12
 * 22 | RoundCondensedSemibold_18 | RoundCondensedSemibold.ttf | 18
 * 23 | RoundCondensedSemibold_24 | RoundCondensedSemibold.ttf | 24
 * 24 | RoundCondensedSemibold_36 | RoundCondensedSemibold.ttf | 36
 * 25 | RoundCondensedSemibold_48 | RoundCondensedSemibold.ttf | 48
 * 26 | RoundCondensedSemibold_60 | RoundCondensedSemibold.ttf | 60
 * 27 | RoundCondensedSemibold_72 | RoundCondensedSemibold.ttf | 72
 * 28 | RoundMedium_12 | RoundMedium.ttf | 12
 * 29 | RoundMedium_18 | RoundMedium.ttf | 18
 * 30 | RoundMedium_24 | RoundMedium.ttf | 24
 * 31 | RoundMedium_36 | RoundMedium.ttf | 36
 * 32 | RoundMedium_48 | RoundMedium.ttf | 48
 * 33 | RoundMedium_60 | RoundMedium.ttf | 60
 * 34 | RoundMedium_72 | RoundMedium.ttf | 72
 * 35 | RoundSemibold_12 | RoundSemibold.ttf | 12
 * 36 | RoundSemibold_18 | RoundSemibold.ttf | 18
 * 37 | RoundSemibold_24 | RoundSemibold.ttf | 24
 * 38 | RoundSemibold_36 | RoundSemibold.ttf | 36
 * 39 | RoundSemibold_48 | RoundSemibold.ttf | 48
 * 40 | RoundSemibold_60 | RoundSemibold.ttf | 60
 * 41 | RoundSemibold_72 | RoundSemibold.ttf | 72
 *
 * Item | Maximum can be created for each screen
 * ----- | -----
 * Text Area | 20
 * Large Button | 8
 * Medium Button | 16
 * Invisible Button | 16
 * Numeric Entry | 1
 * Ethernet Setting | 1
 * Led widget | 1
 * image | 20
 *
 */
int
lcd_addText(IN char * screenName, IN int screenNameLen, IN char * objectName, IN int objectNameLen, IN BYTE alignment,
  IN int xCord, IN int yCord, IN int width, IN int height, IN BYTE fontID, IN BYTE * color, IN int colorLen,
  IN char * label, IN int labelLen, OUT IDTLCDItem * returnItem);

/**
 * Add Image
 *
 *  Adds a image to a selected screen.  Must execute lcd_createScreen first to establish a screen to draw on.
 *  The SDK timeout of the command is set to 6 seconds.
 *
 *    @param screenName Screen name that will be the target of add image
 *    @param screenNameLen  Length of screenName
 *    @param objectName Object name that will be the target of add image
 *    @param objectNameLen  Length of objectName
 *    @param alignment Position for Image
 *      - 0 = Display object at the horizon center of specified y, while x ignored
 *      - 1 = Display object at specified x and y
 *      - 2 = Display object at center of screen, x, y are both ignored
 *      - 3 = Display object at left of the screen of specified y, while x ignored
 *      - 4 = Display object at right of the screen of specified y, while x ignored
 *    @param xCord  x-coordinate for Image, range 0-271
 *    @param yCord  y-coordinate for Image, range 0-479
 *    @param filename Filename of the image. Must be available in device memory.
 *    @param filenameLen  Length of filename
 *    @param returnItem The item includes screen ID, object ID, top-left x-coordinate, top-left y-coordinate, bottom-right x-coordinate, bottom-left y-coordinate, which are all assigned to the created image
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *  Item | Maximum can be created for each screen
 *  ----- | -----
 *  Text Area | 20
 *  Large Button | 8
 *  Medium Button | 16
 *  Invisible Button | 16
 *  Numeric Entry | 1
 *  Ethernet Setting | 1
 *  Led widget | 1
 *  image | 20
 */
int
lcd_addImage(IN char * screenName, IN int screenNameLen, IN char * objectName, IN int objectNameLen, IN BYTE alignment,
  IN int xCord, IN int yCord, IN char * filename, IN int filenameLen, OUT IDTLCDItem * returnItem);

/**
 * Add Video
 *
 *  Adds a video to a selected screen.  Must execute lcd_createScreen first to establish a screen to draw on.
 *  The SDK timeout of the command is set to 6 seconds.
 *
 *    @param screenName Screen name that will be the target of add video
 *    @param screenNameLen  Length of screenName
 *    @param objectName Object name that will be the target of add video
 *    @param objectNameLen  Length of objectName
 *    @param alignment Position for Video
 *      - 0 = Display object at the horizon center of specified y, while x ignored
 *      - 1 = Display object at specified x and y
 *      - 2 = Display object at center of screen, x, y are both ignored
 *      - 3 = Display object at left of the screen of specified y, while x ignored
 *      - 4 = Display object at right of the screen of specified y, while x ignored
 *    @param xCord  x-coordinate for Video, range 0-271
 *    @param yCord  y-coordinate for Video, range 0-479
 *    @param filename Filename of the video. Must be available in the sd card.
 *    @param filenameLen  Length of filename
 *    @param returnItem The item includes screen ID, object ID, top-left x-coordinate, top-left y-coordinate, bottom-right x-coordinate, bottom-left y-coordinate, which are all assigned to the created video
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *  Item | Maximum can be created for each screen
 *  ----- | -----
 *  Text Area | 20
 *  Large Button | 8
 *  Medium Button | 16
 *  Invisible Button | 16
 *  Numeric Entry | 1
 *  Ethernet Setting | 1
 *  Led widget | 1
 *  image | 20
 *  video | 1
 */
int
lcd_addVideo(IN char * screenName, IN int screenNameLen, IN char * objectName, IN int objectNameLen, IN BYTE alignment,
  IN int xCord, IN int yCord, IN char * filename, IN int filenameLen, OUT IDTLCDItem * returnItem);

/**
 * Add Extended Video
 *
 *  Adds a extended video to a selected screen.  Must execute lcd_createScreen first to establish a screen to draw on.
 *  The SDK timeout of the command is set to 6 seconds.
 *
 *    @param screenName Screen name that will be the target of add video
 *    @param screenNameLen  Length of screenName
 *    @param objectName Object name that will be the target of add video
 *    @param objectNameLen  Length of objectName
 *    @param alignment Position for Video
 *      - 0 = Display object at the horizon center of specified y, while x ignored
 *      - 1 = Display object at specified x and y
 *      - 2 = Display object at center of screen, x, y are both ignored
 *      - 3 = Display object at left of the screen of specified y, while x ignored
 *      - 4 = Display object at right of the screen of specified y, while x ignored
 *    @param xCord  x-coordinate for Video, range 0-271
 *    @param yCord  y-coordinate for Video, range 0-479
 *    @param loop  loop the videos when it's done.  0: Do not Loop 1: Loop
 *    @param numVideos  number of video files, range 1-4
 *    @param filenames Filenames of the videos separated by '\0'. Must be available in the sd card.
 *    @param filenamesLen  Length of filenames excluding the last NULL character.  Should be less than 64.
 *    @param returnItem The item includes screen ID, object ID, top-left x-coordinate, top-left y-coordinate, bottom-right x-coordinate, bottom-left y-coordinate, which are all assigned to the created video
 *
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString
 *  Item | Maximum can be created for each screen
 *  ----- | -----
 *  Text Area | 20
 *  Large Button | 8
 *  Medium Button | 16
 *  Invisible Button | 16
 *  Numeric Entry | 1
 *  Ethernet Setting | 1
 *  Led widget | 1
 *  image | 20
 *  video | 1
 */
int
lcd_addExtVideo(IN char * screenName, IN int screenNameLen, IN char * objectName, IN int objectNameLen,
  IN BYTE alignment, IN int xCord, IN int yCord, IN BYTE loop, IN BYTE numVideos, IN char * filenames,
  IN int filenamesLen, OUT IDTLCDItem * returnItem);

/**
 * Clone Screen
 *
 *  Clones an existing screen.
 *    @param screenName  Screen name to clone.
 *    @param screenNameLen  Length of screenName.
 *    @param cloneName  Name of the cloned screen.
 *    @param cloneNameLen  Length of cloneName.
 *    @param cloneID Screen ID of the cloned screen
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString
 */
int
lcd_cloneScreen(IN char * screenName, IN int screenNameLen, IN char * cloneName, IN int cloneNameLen,
  OUT int * cloneID);

/**
 * Update Label
 *
 *  Updates the component label.
 *
 *    @param screenName Screen name that will be the target of update label
 *    @param screenNameLen  Length of screenName.
 *    @param objectName  Identifier of the component
 *    @param objectNameLen  Length of objectName.
 *    @param label Label to show on the component
 *    @param labelLen  Length of label.
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
lcd_updateLabel(IN char * screenName, IN int screenNameLen, IN char * objectName, IN int objectNameLen, IN char * label,
  IN int labelLen);

/**
 * Update Color
 *
 *  Updates the component color, or updates the LED colors if specifying LCD component
 *
 *   @param screenName Screen name that will be the target of update color
 *   @param screenNameLen  Length of screenName.
 *   @param objectName  Identifier of the component
 *   @param objectNameLen  Length of objectName.
 *   @param color \r\n
 *     Non LCD Widget: Four bytes for color, example, Blue = 0xFF000000, Black = 0x00000000
 *      - Byte 0 = B
 *      - Byte 1 = G
 *      - Byte 2 = R
 *      - Byte 3 = Reserved
 *     LCD Widget: Four bytes for  LED color, byte 0 = LED 0, byte 1 = LED 1, byte 2 = LED2, byte 3 = LED 3
 *      - Value 0 = LED OFF
 *      - Value 1 = LED Green
 *      - Value 2 = LED Yellow
 *      - Value 3 = LED Red
 *   @param colorLen  Length of color.
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
lcd_updateColor(IN char * screenName, IN int screenNameLen, IN char * objectName, IN int objectNameLen, IN BYTE * color,
  IN int colorLen);

/**
 * Update Position
 *
 *  Updates the component position.
 *
 *    @param screenName Screen Name that will be the target of update position
 *    @param screenNameLen  Length of screenName.
 *    @param objectName Identifier of the component
 *    @param objectNameLen  Length of objectName.
 *    @param alignment  Alignment for the target
 *      - 0 = Display object at the horizon center of specified y, while x ignored
 *      - 1 = Display object at specified x andy
 *      - 2 = Display object at center of screen, x, y are both ignored
 *      - 3 = Display object at left of the screen of specified y, while x ignored
 *      - 4 = Display object at right of the screen of specified y, while x ignored
 *    @param new_xCord  x-coordinate for Text, range 0-271
 *    @param new_yCord  y-coordinate for Text, range 0-479
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
lcd_updatePosition(IN char * screenName, IN int screenNameLen, IN char * objectName, IN int objectNameLen,
  IN BYTE alignment, IN int new_xCord, IN int new_yCord);

/**
 * Removed Item
 *
 *  Removes a component.
 *
 *    @param screenName Screen name where to remove the target from.
 *    @param screenNameLen  Length of screenName.
 *    @param objectName  Identifier of the component
 *    @param objectNameLen  Length of objectName.
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
lcd_removeItem(IN char * screenName, IN int screenNameLen, IN char * objectName, IN int objectNameLen);

/**
 * Store Screen Info
 *
 *  Store all current screen information from RAM to flash
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
lcd_storeScreenInfo();

/**
 * Load Screen Info
 *
 *  Load all current screen information from flash to RAM
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
lcd_loadScreenInfo();

/**
 * Clear Screen Info
 *
 *  Clear all current screen information in RAM and flash. And then show'power-on screen'
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
lcd_clearScreenInfo();

/**
 * Clear Display
 *  Command to clear the display screen on the reader. It returns the
 *  display to the currently defined background color and terminates
 *  all events
 *
 * @param control for L80 and L100 only.
 *     0:First Line
 *             1:Second Line
 *             2:Third Line
 *             3:Fourth Line
 *             0xFF: All Screen
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
lcd_clearDisplay(IN BYTE control);

/**
 * Display Message on Line
 *
 *      Displays a message on a display line.
 *        @param lineNumber  Line number to display message on  (1-4)
 *        @param message Message to display
 *        @param messageLen length of message
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString
 *
 */
int
lcd_displayMessage(int lineNumber,
  char *               message,
  int                  messageLen);

/**
 * Get All Screens
 *
 *  Get all created screens' name
 *
 *    @param screenNumbers Number of created screens
 *    @param screenInfo Array of all created screens
 *        each element includes
 *           -screenID of a created screen
 *           -screenName of a created screen
 *           -screenNameLen of screenName
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
lcd_getAllScreens(IN_OUT int * screenNumbers, OUT IDTScreenInfo * screenInfo);

/**
 * Get All Objects on Screen
 *
 *  Get all created objects' name on certain screen
 *
 *    @param screenName Screen name to get all objects
 *    @param screenNameLen  Length of screenName
 *    @param objectNumbers Number of created objects
 *    @param returnObjects Array of all created objects
 *       each element includes
 *           -objectID of a created object
 *           -objectName of a created object
 *           -objectNameLen of objectName
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
lcd_getAllObjects(IN char * screenName, IN int screenNameLen, IN_OUT int * objectNumbers,
  OUT IDTObjectInfo * objectInfo);

/**
 * Queery Screen by Name
 *
 *  Check if the given screen exists or not
 *
 *    @param screenName Name of the checked screen
 *    @param screenNameLen  Length of screenName
 *    @param result the checking result
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
lcd_queryScreenbyName(IN char * screenName, IN int screenNameLen, OUT int * result);

/**
 * Link UI with Transaction Message ID
 *
 *  Link an existing Customer UI ID with a specified transaction message ID. During transaction, the linked System UI will be replaced by the linked Customer UI
 *
 *    @param MessageId Transaction Message ID: Refer to Doc “EMV Display Message ID Assignment” Selection by Customer
 *    @param screenName Name of the screen (No longer than 32 bytes including the NULL character)
 *    @param screenNameLen  Length of screenName
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
lcd_linkUIWithTransactionMessageId(IN BYTE MessageId, IN char * screenName, IN int screenNameLen);

/**
 * Queery Object by Name
 *
 *  Check if the given object exists or not. If exists, return all screen names which contains the object of the given object name
 *
 *    @param objectName Name of the checked object
 *    @param objectNameLen Length of objectName
 *    @param objectNumbers Number of the checked object
 *    @param screenInfo Array of all the screen names that contain the object
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
lcd_queryObjectbyName(IN char * objectName, IN int objectNameLen, IN_OUT int * objectNumbers,
  OUT IDTScreenInfo * screenInfo);

/**
 * Queery Screen by ID
 *
 *  Check if the given screen exists or not
 *
 *    @param screenID ID of the checked screen
 *    @param result the checking result
 *    @param screenName Name of the checked screen
 *    @param screenNameLen Length of screenName
 *    @param ip Optional: IP address to execute command on (for IP connected devices)
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
lcd_queryScreenbyID(IN int screenID, OUT int * result, OUT int * screenName, IN_OUT int * screenNameLen);

/**
 * Queery Object by ID
 *
 *  Check if the given object exists or not. If exists, return all screen names which contains the object of the given object ID
 *
 *    @param objectID ID of the checked object
 *    @param objectNumbers Number of the checked object
 *    @param screenInfo screen names containing the item
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
lcd_queryObjectbyID(IN int objectID, OUT int * objectNumbers, OUT IDTScreenInfo * screenInfo);

/**
 * Set Backlight
 *
 *  Set backlight percentage. If the percent > 100, it will be rejected. If 0 < percent < 10, backlight percent will be set to 10.
 *    @param isBacklightOn 0: backlight off 1: backlight on
 *    @param backlightVal Backlight percent value to be sat
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
lcd_setBacklight(IN BYTE isBacklightOn, IN BYTE backlightVal);

/**
 * Disable MSR Swipe
 * Cancels MSR swipe request.
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
msr_cancelMSRSwipe();

/**
 * Start MSR Swipe
 *
 * Enables MSR, waiting for swipe to occur. Allows track
 * selection. Returns IDTMSRData instance to swipeMSRData()
 * @param timeout Swipe Timeout Value
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 * Note: if auto poll mode is on, it will return command not allowed error
 */
int
msr_startMSRSwipe(IN int _timeout);

/**
 * executeTransaction
 *  Requests 3 sets of public keys: encrypting Keys, signing/validating keys, signing/validating 3rd party apps
 *
 * @param data WorldPay data object
 * @param wpCallback WorldPay callback
 * @param requestOnly
 *  - 0 = send transaction and return response,
 *  - 1 = send transaction and return both request and response,
 *  - 2 = do not send transaction, instead return request
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
executeTransaction(WorldPayData * data, pWP_callBack wpCallback, int requestOnly);

/**
 * forwardTransaction
 *  Send the saved data to WorldPay and complete the transaction.
 *
 * @param wpCallback WPCallback is the callback to send the results.  Should the the same as executeTransaction callback.
 * @param forwardID = ID, which could be either unique ID or Memo.
 * @param forwardIDLen The length of forwardID.
 * @param password = password.  If null/blank, no password.
 * @param passwordLen The length of passwordLen.
 * @param bypassProcess = true means read the file from disk, but don't send to WorldPay, then delete the transaction as if you did send to WorldPay.  The purpose of this is to allow them to delete transactions from the storage without sending to WorldPay.
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
forwardTransaction(IN pWP_callBack wpCallback, IN char *forwardID, IN int forwardIDLen, IN char *password, IN int passwordLen, IN int bypassProcessing);

/**
 * Cancels WorldPay transaction.
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
cancelWorldPay();

/**
 * Parser the MSR data from the buffer into IDTMSRData structure
 * @param resData MSR card data buffer
 * @param resLen the length of resData
 * @param cardData the parser result with IDTMSTData structure
 */
void
parseMSRData(IN BYTE * resData,
  IN int               resLen,
  IN_OUT IDTMSRData *  cardData);

/**
 * Capture PIN
 *
 *  @param timeout Timeout, in seconds. Value of 0 will use system timeout, if any
 *  @param type PAN and Key Type
 * - 00h = MKSK to encrypt PIN, Internal PAN (from MSR)
 * - 01h = DUKPT to encrypt PIN, Internal PAN (from MSR)
 * - 10h = MKSK to encrypt PIN, External Plaintext PAN
 * - 11h = DUKPT to encrypt PIN, External Plaintext PAN
 * - 20h = MKSK to encrypt PIN, External Ciphertext PAN
 * - 21h = DUKPT to encrypt PIN, External Ciphertext PAN
 *
 * @param PAN Personal Account Number (if internal, value is 0)
 * @param PANLen Length of PAN
 * @param minPIN Minimum PIN Length
 * @param maxPIN Maximum PIN Length
 * @param message LCD Message
 * @param messageLen Length of message
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
pin_capturePin(IN int timeout, IN int type, IN char * PAN, IN int PANLen, IN int minPIN, IN int maxPIN,
  IN char * message, IN int messageLen);

/** * Capture PIN Ext
 *
 * @param type PAN and Key Type
 * - 00h: MKSK to encrypt PIN, Internal PAN (from MSR or Manual PAN Entry or Contactless EMV Transaction)
 * - 01h: DUKPT to encrypt PIN, Internal PAN (from MSR or Manual PAN Entry or Contactless EMV Transaction)
 * - 10h: MKSK to encrypt PIN, External Plaintext PAN
 * - 11h:  DUKPT to encrypt PIN, External Plaintext PAN
 * - 20h: MKSK to encrypt PIN, External Ciphertext PAN (for PIN pad only)
 * - 21h:  DUKPT to encrypt PIN, External Ciphertext PAN (for PIN pad only)
 * - 80h:  MKSK to encrypt PIN, Internal PAN, Verify PIN (from MSR or Manual PAN Entry or Contactless EMV Transaction)
 * - 81h:  DUKPT to encrypt PIN, Internal PAN, Verify PIN (from MSR or Manual PAN Entry or Contactless EMV Transaction)
 * - 90h: MKSK to encrypt PIN, External Plaintext PAN, Verify PIN
 * - 91h: DUKPT to encrypt PIN, External Plaintext PAN, Verify PIN
 *
 * @param PAN Personal Account Number (if internal, value is 0)
 * @param PANLen Length of PAN
 * @param minPIN Minimum PIN Length
 * @param maxPIN Maximum PIN Length
 * @param message LCD Message Up to 2 lines of text, each line 1-16, separated by 0x00
 * @param messageLen Length of 1st scenario LCD message, valid in 00h~21h (0~33).If no LCD message input, length is 00h, and display default msg “PLEASE ENTER PIN”
 * @param verify LCD Message Up to 2 lines of text, each line 1-16, separated by 0x00
 * @param verifyLen Length of 2nd Scenario LCD message.valid in 00h~21h (0~33).This field is present only when PAN and Key Type has Verify PIN.If no LCD message input, length is 00h, and display default msg “ ENTER PIN AGAIN”
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
pin_capturePinExt(IN int type, IN char * PAN, IN int PANLen, IN int minPIN, IN int maxPIN, IN char * message,
  IN int messageLen, IN char * verify, IN int verifyLen);

/**
 * Capture Numeric Input
 *
 *
 * @param timeout Timeout, in seconds. Value of 0 will use system timeout, if any
 * @param function
 *    - 0x00 = Plaintext Input
 *    - 0x01 = Masked Input
 *    - 0x02 = Delayed Masking Input
 *    - 0x10 = Plaintext Input + MSR Active
 *    - 0x11 = Masked Input + MSR Active
 *    - 0x12 = Delayed Masking Input + MSR Active
 * @param minLen Minimum input Length
 * @param maxLen Maximum input Length
 * @param line1 Line 1 of LCD Message - 16 chars max
 * @param line1Len Length of line1
 * @param line2 Line 2 of LCD Message - 16 chars max
 * @param line2Len Length of line2
 * @param signature Display message signed by Numeric Private Key using RSAPSS algorithm:
 *   1. Calculate 32 bytes Hash for ��<Display Flag><Key Max Length>< Key Min Length><Plaintext Display Message>��
 *   2. Using RSAPSS algorithm calculate the Hash to be 256 bytes Raw Data
 *   3. Using Numeric Private Key to sign the Raw Data to be 256 bytes signature
 * @param signatureLen Length of signature
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
pin_promptForNumericKeyWithSwipe(IN int timeout, IN BYTE function, IN int minLen, IN int maxLen, IN char * line1,
  IN int line1Len, IN char * line2, IN int line2Len, BYTE * signature,
  IN int signatureLen);

/**
 * Capture Numeric Input
 *
 *
 * @param timeout Timeout, in seconds. Value of 0 will use system timeout, if any
 * @param maskInput
 *   - 0 = Display numeric for numeric key on LCD
 *   - 1 = Display * for numeric key on LCD
 * @param minLen Minimum input Length
 * @param maxLen Maximum input Length
 * @param message Plaintext Display Message. - 16 chars max
 * @param messageLen Length of message
 * @param signature Display message signed by Numeric Private Key using RSAPSS algorithm:
 *  1. Calculate 32 bytes Hash for <Display Flag><Key Max Length>< Key Min Length><Plaintext Display Message>
 *  2. Using RSAPSS algorithm calculate the Hash to be 256 bytes Raw Data
 *  3. Using Numeric Private Key to sign the Raw Data to be 256 bytes signature
 * @param signatureLen Length of signature
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
pin_promptForNumericKey(IN int timeout, IN int maskInput, IN int minLen, IN int maxLen, IN char * message,
  IN int messageLen, BYTE * signature, IN int signatureLen);

/**
 * Get PIN Input from Prompt
 *            Results returned to PIN Callback.  If successful function key capture, data is returned as
 *            IDTPINData.keyString.
 *
 *
 *            @param mask 0 = no masking, 1 = Display "*" for numeric key according to Pre-Cleartext and Post-Cleartext display
 *          @param preClearText   Range 0-6 Characters to mask at start of text if masking is on;
 *          @param postClearText  Range 0-6 Characters to mask at end of text if masking is on;
 *          @param minLen  Minimum number of digits required to input
 *          @param maxLen  Maximum number of digits allowed to be input
 *          @param lang  Valid values;
 *          “EN” is English display message
 *          “JP” is Japanese display message
 *          “ES” is Spanish display message
 *          “FR” is French display message
 *          “ZH” is Chinese display message
 *          “PT” is Portuguese display message
 *          @param promptID  Valid values:
 *          0x00: Enter Phone Number
 *          0x01: Enter IP Address
 *          0x02: Enter Subnet Mask
 *          0x03: Enter Default Gateway
 *          0x04: Enter Odometer Reading/Mileage
 *          0x05: Enter Employee ID
 *          0x06: Enter Password for Default Factory Setting
 *          0x07: Enter Email Address (Full keyboard)
 *          @param defaultResponse  Default String on input field
 *          @param defaultResponseLen  Length of defaultResponse
 *          @param timeout  Timeout, in seconds
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
pin_inputFromPrompt(BYTE mask, BYTE preClearText, BYTE postClearText, int minLen, int maxLen, char * lang,
  BYTE promptID, char * defaultResponse, int defaultResponseLen, int timeout);

/**
 * Get Pan
 *
 * prompt the user to manually enter a card PAN and Expiry Date (and optionally CSC) from the keypad and return it to the POS.
 * @param csc Request CSS
 *   @param expDate Request Expiration Date
 *   @param ADR Request Address
 *   @param ZIP Request Zip
 *   @param mod10CK Validate entered PAN passes MOD-10 checking before accepting
 *   @param timeout Number of seconds that the reader waits for the data entry session to complete, stored as a big-endian number. 0 = no timeout
 *   @param encPANOnly Output only encrypted PAN
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
pin_getPanEntry(IN int csc, IN int expDate, IN int ADR, IN int ZIP, IN int mod10CK, IN int timeout, IN int encPANOnly);

/**
 * Cancel PIN Entry
 *
 * Cancels PIN entry request
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
pin_cancelPINEntry();

/**
 * Set Key Values
 *
 * Set return key values on or off
 *
 * @param mode  On: 1, Off: 0
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
pin_setKeyValues(int mode);

# ifdef __cplusplus
}
# endif


#endif // ifndef __LIBIDT_NEO2_H___

/*! \file libIDT_NEO2.h
 * \brief NEO2 API.
 *
 * NEO2 Global API methods.
 */

/*! \def IN
 * INPUT parameter.
 */

/*! \def OUT
 * OUTPUT parameter.
 */

/*! \def IN_OUT
 * INPUT / OUTPUT PARAMETER.
 */

/*! \def _DATA_BUF_LEN
 * DATA BUFFER LENGTH
 */
