#ifndef _IDTECHSDK_H_
# define _IDTECHSDK_H_

# include "IDTDef.h"
# include "libIDT_Device_str.h"

# ifdef __cplusplus
extern "C" {
# endif /* __cplusplus */

# define MAX_TAG_LENGTH      64
# define MAX_TAG_DATA_LENGTH 1024

/*
 *  static const char *openingDataElementTag = "<DataElement>";
 *  static const char *closingDataElementTag = "</DataElement>";
 *  static const char *fieldUsageTagName = "FieldUsage";
 *  static const char *dataTagName = "Data";
 *  static const char *openingDataTag = "<Data>";
 *  static const char *closingDataTag = "</Data>";
 *  static const char *dataFieldTagName = "DataField";
 *  static const char *openingDataFieldTag = "<DataField>";
 *  static const char *closingDataFieldTag = "</DataField>";
 *  static const char *ISO8583_1987TagName = "ISO8583_1987_Message";
 *  static const char *ISO8583_1993TagName = "ISO8583_1993_Message";
 *  static const char *ISO8583_2003TagName = "ISO8583_2003_Message";
 *
 *  typedef struct {
 *      DL_ISO8583_FIELD_DEF *fieldDef;
 *      DL_ISO8583_MSG_FIELD msgField;
 *  } DataElement;
 *
 *  typedef struct {
 *      DL_UINT8 openingTag[MAX_TAG_LENGTH];
 *      DL_UINT8 closingTag[MAX_TAG_LENGTH];
 *
 *      DataElement dataElements[kDL_ISO8583_MAX_FIELD_IDX + 1];
 *      DL_UINT8 dataElementsLength;
 *  } ISO8583XMLStructure;
 */
int
config_setCmdTimeOutDuration(int millisecond);
int
config_getSerialNumber(char * sNumber);
int
config_getSerialNumber_Len(char * sNumber, IN_OUT int * sNumberLen);
int
config_getModelNumber(char * sNumber);
int
config_getModelNumber_Len(char * sNumber, IN_OUT int * sNumberLen);

void
floatToString(float n, char * res);
int
intToStr(int x, char str[], int d);

int
isNGA(int deviceType);
char *
getDeviceType(int deviceType);
int
device_isConnected();
int
device_getCurrentDeviceType();
int
DoIDGCMD_str(BYTE command, BYTE subCommand, BYTE * data, int dataLength, BYTE * resp, int * respLen, int ignore);

/**
 * To register the ctls callback function to get the MSR card data.
 */
void
ctls_registerCallBk(pMSR_callBack pCTLSf);

/**
 * To register the ctls callback function to get the MSR card data pointer.
 */
void
ctls_registerCallBkp(pMSR_callBackp pCTLSf);

/**
 * Internal use only.
 */
int
processCTLSData(BYTE * retData, int retDataLen, IDTMSRData * cardData);

/**
 * Start CTLS Transaction Request
 *
 * Authorizes the CTLS transaction for an ICC card
 *
 * The tags will be returned in the callback routine.
 *
 * @param amount Transaction amount value (tag value 9F02)
 * - SEE IMPORTANT NOTE BELOW
 * @param amtOther Other amount value, if any (tag value 9F03)
 * - SEE IMPORTANT NOTE BELOW
 * @param type Transaction type (tag value 9C).
 * @param timeout Timeout value in seconds.
 * @param tags Any other tags to be included in the request.  Passed as
 * TLV stream.  Example, tag 9F02 with amount 0x000000000100 would be
 * 0x9F0206000000000100.  If tags 9F02 (amount),9F03 (other amount), or 9C
 * (transaction type) are included, they will take priority over these
 * values supplied as individual parameters to this method.
 *
 * @param tagsLen The length of tags data buffer.
 *
 * >>>>>IMPORTANT: parameters for amount and amtOther MUST BE PASSED AS
 *    A DOUBLE VALUE WITH DECIMAL POINT.  Example, do not pass 1, but
 *    instead pass 1.0 or 1.00. Otherwise, results will be
 *    unpredictable
 *
 * @return RETURN_CODE: Values can be parsed with
 * device_getIDGStatusCodeString() Note: if auto poll is on, it will
 * returm the error IDG_P2_STATUS_CODE_COMMAND_NOT_ALLOWED
 *
 * NOTE ON APPLEPAY VAS:
 * To enable ApplePay VAS, first a merchant record must be defined in
 * one of the six available index positions (1-6) using
 * device_setMerchantRecord, then container tag FFEE06 must be sent as
 * part of the additional tags parameter of ctls_startTransaction.
 * Tag FFEE06 must contain tag 9F26 and 9F22, and can optionanally
 * contain tags 9F2B and DF01.  Example
 * FFEE06189F220201009F2604000000009F2B050100000000DF010101
 * 9F22 = two bytes = ApplePay Terminal Applicaiton Version Number.
 *        Hard defined as 0100 for now. (required)
 * 9F26 = four bytes = ApplePay Terminal Capabilities Information (required)
 *  - Byte 1 = RFU
 *  - Byte 2 = Terminal Type
 *  - - Bit 8 = VAS Support     (1=on, 0 = off)
 *  - - Bit 7 = Touch ID Required  (1=on, 0 = off)
 *  - - Bit 6 = RFU
 *  - - Bit 5 = RFU
 *  - - Bit 1,2,3,4
 *  - - - 0 = Payment Terminal
 *  - - - 1 = Transit Terminal
 *  - - - 2 = Access Terminal
 *  - - - 3 = Wireless Handoff Terminal
 *  - - - 4 = App Handoff Terminal
 *  - - - 15 = Other Terminal
 *  - Byte 3 = RFU
 *  - Byte 4 = Terminal Mode
 *  - - 0 = ApplePay VAS OR ApplePay
 *  - - 1 = ApplePay VAS AND ApplePay
 *  - - 2 = ApplePay VAS ONLY
 *  - - 3 = ApplePay ONLY
 *  9F2B = 5 bytes = ApplePay VAS Filter.  Each byte filters for that
 *         specific merchant index  (optional)
 *  DF01 = 1 byte = ApplePay VAS Protocol.  (optional)
 *  - - Bit 1 : 1 = URL VAS, 0 = Full VAS
 *  - - Bit 2 : 1 = VAS Beeps, 0 = No VAS Beeps
 *  - - Bit 3 : 1 = Silent Comm Error, 2 = EMEA Comm Error
 *  - - Bit 4-8 : RFU
 *
 */
int
ctls_startTransaction(float amount, float amtOther, int type, const int _timeout, BYTE * tags, int tagsLen);

/**
 * Cancel EMV Transaction
 *
 * Cancels the currently executing EMV transaction.
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
ctls_cancelTransaction();

/**
 * Retrieve Application Data by AID
 *
 * Retrieves the Application Data as specified by the AID name passed as
 * a parameter.
 *
 * @param AID Name of ApplicationID. Must be between 5 and 16 bytes
 * @param AIDLen the length of AID data buffer.
 * @param tlv  The TLV elements of the requested AID
 * @param tlvLen the length of tlv data buffer.
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
ctls_retrieveApplicationData(IN BYTE * AID,
  IN int                               AIDLen,
  OUT BYTE *                           tlv,
  IN_OUT int *                         tlvLen);

/**
 * Set Application Data by AID
 *
 * Sets the Application Data for CTLS as specified by TLV data
 *
 * @param tlv  Application data in TLV format
 *      The first tag of the TLV data must be the group number (FFE4).
 *      The second tag of the TLV data must be the AID (9F06)
 *
 *      Example valid TLV, for Group #2, AID a0000000035010:
 *      "ffe401029f0607a0000000051010ffe10101ffe50110ffe30114ffe20106"
 * @param tlvLen the length of tlv data buffer
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
ctls_setApplicationData(IN BYTE * tlv,
  IN int                          tlvLen);

/**
 * Remove Application Data by AID
 * Removes the Application Data for CTLS as specified by the AID name
 * passed as a parameter
 *
 * @param AID Name of ApplicationID Must be between 5 and 16 bytes
 *
 * @param AIDLen the length of AID data buffer
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
ctls_removeApplicationData(IN BYTE * AID,
  IN int                             AIDLen);

/**
 * Remove All Application Data
 *
 * Removes all the Application Data
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 **/
int
ctls_removeAllApplicationData();

/**
 * Retrieve AID list
 *
 * Returns all the AID names installed on the terminal for CTLS. .
 *
 * @param AIDList  array of AID name byte arrays
 * @param AIDListLen  the length of AIDList array buffer
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
ctls_retrieveAIDList(OUT BYTE * AIDList,
  IN_OUT int *                  AIDListLen);

/**
 * Retrieve Terminal Data
 *
 *
 * Retrieves the Terminal Data for CTLS. This is configuration group 0
 * (Tag FFEE - > FFEE0100).  The terminal data can also be retrieved
 * by ctls_getConfigurationGroup(0).
 *
 * @param tlv Response returned as a TLV
 * @param tlvLen the length of tlv data buffer
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
ctls_retrieveTerminalData(OUT BYTE * tlv,
  IN_OUT int *                       tlvLen);

/**
 * Set Terminal Data
 *
 * Sets the Terminal Data for CTLS as specified by the TLV.  The first
 *      TLV must be Configuration Group Number (Tag FFE4).  The
 *      terminal global data is group 0, so the first TLV would be
 *      FFE40100.  Other groups can be defined using this method (1 or
 *      greater), and those can be retrieved with
 *      ctls_getConfigurationGroup(int group), and deleted with
 *      ctls_removeConfigurationGroup(int group).  You cannot delete
 *      group 0.
 *
 * @param tlv TerminalData configuration file
 * @param tlvLen the length of tlv data buffer
 * @retval RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
ctls_setTerminalData(IN BYTE * tlv,
  IN int                       tlvLen);

/**
 * Retrieve Certificate Authority Public Key
 *
 *
 * Retrieves the CAPK for CTLS as specified by the RID/Index passed as a
 * parameter.
 *
 * @param capk 6 bytes CAPK = 5 bytes RID + 1 byte Index
 * @param capkLen the length of capk data buffer
 * @param key Response returned as a CAKey format:
 *  [5 bytes RID][1 byte Index][1 byte Hash Algorithm][1 byte Encryption Algorithm]
 *  [20 bytes HashValue][4 bytes Public Key Exponent][2 bytes Modulus Length][Variable bytes Modulus]
 *  Where:
 *   - Hash Algorithm: The only algorithm supported is SHA-1.The value is set to 0x01
 *   - Encryption Algorithm: The encryption algorithm in which this key is used. Currently support only one type: RSA. The value is set to 0x01.
 *   - HashValue: Which is calculated using SHA-1 over the following fields: RID & Index & Modulus & Exponent
 *   - Public Key Exponent: Actually, the real length of the exponent is either one byte or 3 bytes. It can have two values: 3 (Format is 0x00 00 00 03), or  65537 (Format is 0x00 01 00 01)
 *   - Modulus Length: LenL LenH Indicated the length of the next field.
 *   - Modulus: This is the modulus field of the public key. Its length is specified in the field above.
 * @param keyLen the length of key data buffer
 * * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
ctls_retrieveCAPK(IN BYTE * capk,
  IN int                    capkLen,
  OUT BYTE *                key,
  IN_OUT int *              keyLen);

/**
 * Set Certificate Authority Public Key
 *
 * Sets the CAPK as specified by the CAKey structure
 *
 * @param capk CAKey format:
 *  [5 bytes RID][1 byte Index][1 byte Hash Algorithm][1 byte Encryption Algorithm][20 bytes HashValue][4 bytes Public Key Exponent][2 bytes Modulus Length][Variable bytes Modulus]
 *  Where:
 *   - Hash Algorithm: The only algorithm supported is SHA-1.The value is set to 0x01
 *   - Encryption Algorithm: The encryption algorithm in which this key is used. Currently support only one type: RSA. The value is set to 0x01.
 *   - HashValue: Which is calculated using SHA-1 over the following fields: RID & Index & Modulus & Exponent
 *   - Public Key Exponent: Actually, the real length of the exponent is either one byte or 3 bytes. It can have two values: 3 (Format is 0x00 00 00 03), or  65537 (Format is 0x00 01 00 01)
 *   - Modulus Length: LenL LenH Indicated the length of the next field.
 *   - Modulus: This is the modulus field of the public key. Its length is specified in the field above.
 * @param capkLen the length of capk data buffer
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
ctls_setCAPK(IN BYTE * capk,
  IN int               capkLen);
int
ctls_retrieveCAPKRIDList(BYTE * keys, int * keysLen);
int
ctls_retrieveCAPKRIDIndexes(BYTE * RID, int RIDLen, BYTE * tlv, int * tlvLen);

/**
 * Remove Certificate Authority Public Key
 *
 * Removes the CAPK as specified by the RID/Index
 *
 * @param capk 6 byte CAPK =  5 bytes RID + 1 byte INDEX
 * @param capkLen the length of capk data buffer
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
ctls_removeCAPK(IN BYTE * capk,
  IN int                  capkLen);

/**
 * Remove All Certificate Authority Public Key
 *
 * Removes all the CAPK
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
ctls_removeAllCAPK();

/**
 * Retrieve the Certificate Authority Public Key list
 *
 * Returns all the CAPK RID and Index installed on the terminal.
 *
 * @param keys [key1][key2]...[keyn], each key 6 bytes where
 *  key = 5 bytes RID + 1 byte index
 * @param keysLen the length of keys data buffer
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
ctls_retrieveCAPKList(OUT BYTE * keys,
  IN_OUT int *                   keysLen);

/**
 * Set Configuration Group
 *
 * Sets the Configuration Group for CTLS as specified by the TLV data
 *
 * @param tlv  Configuration Group Data in TLV format
 *        The first tag of the TLV data must be the group number (DFEE2D).
 *        A second tag must exist
 * @param tlvLen the length of tlv data buffer
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
ctls_setConfigurationGroup(IN BYTE * tlv,
  IN int                             tlvLen);

/**
 * Get Configuration Group
 *
 * Retrieves the Configuration for the specified Group.
 *
 * @param group Configuration Group
 * @param tlv return data
 * @param tlvLen the length of tlv data buffer
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
ctls_getConfigurationGroup(IN int group,
  OUT BYTE *                      tlv,
  OUT int *                       tlvLen);

/**
 * Retrieve All Configuration Groups
 *
 * Returns all the Configuration Groups installed on the terminal for CTLS
 *
 * @param tlv  The TLV elements data
 * @param tlvLen the length of tlv data buffer.
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
ctls_getAllConfigurationGroups(OUT BYTE * tlv,
  IN_OUT int *                            tlvLen);

/**
 * Remove Configuration Group
 *
 * Removes the Configuration as specified by the Group.  Must not by group 0
 *
 * @param group Configuration Group
 * @retval RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
ctls_removeConfigurationGroup(int group);


/**
 * Polls device for EMV Kernel Version
 *
 * @param version Response returned of Kernel Version
 *
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString()
 *
 */
int
emv_getEMVKernelVersion(OUT char * version);
int
emv_getEMVKernelVersion_Len(OUT char * version, IN_OUT int * versionLen);

/**
 * Get EMV Kernel check value info
 *
 * @param checkValue Response returned of Kernel check value info
 * @param checkValueLen the length of checkValue
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString
 */
int
emv_getEMVKernelCheckValue(OUT BYTE * checkValue,
  IN_OUT int *                        checkValueLen);

/**
 * Get EMV Kernel configuration check value info
 *
 * @param checkValue Response returned of Kernel configuration check value info
 * @param checkValueLen the length of checkValue
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString
 *
 */
int
emv_getEMVConfigurationCheckValue(OUT BYTE * checkValue,
  IN_OUT int *                               checkValueLen);

/**
 * Set Application Data by AID
 *
 * Sets the Application Data as specified by the application name and TLV data
 *
 * @param name Application name, 10-32 ASCII hex characters representing
 * 5-16 bytes  Example "a0000000031010"
 * @param nameLen the length of name data buffer of Application name,
 * @param tlv  Application data in TLV format
 * @param tlvLen the length of tlv data buffer
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString()
 */
int
emv_setApplicationData(IN BYTE * name,
  IN int                         nameLen,
  IN BYTE *                      tlv,
  IN int                         tlvLen);

/**
 * Set Application Data by TLV
 *
 * Sets the Application Data as specified by the TLV data
 * @param tlv  Application data in TLV format
 *              The first tag of the TLV data must be the group number (DFEE2D).
 *              The second tag of the TLV data must be the AID (9F06)
 *              Example valid TLV, for Group #2, AID a0000000035010:
 *              "dfee2d01029f0607a0000000051010ffe10101ffe50110ffe30114ffe20106"
 * @param tlvLen the length of tlv data buffer
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString()
 */
int
emv_setApplicationDataTLV(IN BYTE * tlv,
  IN int                            tlvLen);

/**
 * Remove Application Data by AID
 * Removes the Application Data for CTLS as specified by the AID name passed
 * as a parameter
 * @param AID Name of ApplicationID Must be between 5 and 16 bytes
 * @param AIDLen the length of AID data buffer
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_removeApplicationData(IN BYTE * AID,
  IN int                            AIDLen);

/**
 * Remove all Application Data
 */
int
emv_removeAllApplicationData();

/**
 * Retrieve Terminal Data
 *
 * Retrieves the Terminal Data for CTLS. This is configuration group 0
 * (Tag FFEE - > FFEE0100).  The terminal data can also be retrieved
 * by ctls_getConfigurationGroup(0).
 * @param tlv Response returned as a TLV
 * @param tlvLen the length of tlv data buffer
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_retrieveTerminalData(OUT BYTE * tlv,
  IN_OUT int *                      tlvLen);

/**
 * Set Terminal Data
 *
 * Sets the Terminal Data for CTLS as specified by the TLV.  The first
 * TLV must be Configuration Group Number (Tag FFE4).  The terminal
 * global data is group 0, so the first TLV would be FFE40100.  Other
 * groups can be defined using this method (1 or greater), and those
 * can be retrieved with emv_getConfigurationGroup(int group), and
 * deleted with emv_removeConfigurationGroup(int group).  You cannot
 * delete group 0.
 *
 * @param tlv TerminalData configuration file
 * @param tlvLen the length of tlv data buffer
 * @retval RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_setTerminalData(IN BYTE * tlv,
  IN int                      tlvLen);

/**
 * Sets the terminal major configuration in ICS .
 * @param configuration A configuration value, range 1-5
 * - 1 = 1C
 * - 2 = 2C
 * - 3 = 3C
 * - 4 = 4C
 * - 5 = 5C
 *
 *  @return RETURN_CODE: Values can be parsed with device_getResponseCodeString()
 */
int
emv_setTerminalMajorConfiguration(IN int configuration);

/**
 * Remove Terminal Data
 */
int
emv_removeTerminalData();

/**
 * Retrieve AID list
 *
 * Returns all the AID names installed on the terminal for CTLS. .
 *
 * @param AIDList  array of AID name byte arrays
 * @param AIDListLen  the length of AIDList array buffer
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_retrieveAIDList(OUT BYTE * AIDList,
  IN_OUT int *                 AIDListLen);

/**
 * Retrieve Certificate Authority Public Key
 *
 * Retrieves the CAPK for CTLS as specified by the RID/Index passed as a
 * parameter.
 *
 * @param capk 6 bytes CAPK = 5 bytes RID + 1 byte Index
 * @param capkLen the length of capk data buffer
 * @param key Response returned as a CAKey format:
 *  [5 bytes RID][1 byte Index][1 byte Hash Algorithm][1 byte Encryption Algorithm]
 *  [20 bytes HashValue][4 bytes Public Key Exponent][2 bytes Modulus Length][Variable bytes Modulus]
 *  Where:
 *   - Hash Algorithm: The only algorithm supported is SHA-1.The value is set to 0x01
 *   - Encryption Algorithm: The encryption algorithm in which this key is used. Currently support only one type: RSA. The value is set to 0x01.
 *   - HashValue: Which is calculated using SHA-1 over the following fields: RID & Index & Modulus & Exponent
 *   - Public Key Exponent: Actually, the real length of the exponent is either one byte or 3 bytes. It can have two values: 3 (Format is 0x00 00 00 03), or  65537 (Format is 0x00 01 00 01)
 *   - Modulus Length: LenL LenH Indicated the length of the next field.
 *   - Modulus: This is the modulus field of the public key. Its length is specified in the field above.
 * @param keyLen the length of key data buffer
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_retrieveCAPK(IN BYTE * capk,
  IN int                   capkLen,
  OUT BYTE *               key,
  IN_OUT int *             keyLen);

/**
 * Set Certificate Authority Public Key
 *
 * Sets the CAPK as specified by the CAKey structure
 *
 * @param capk CAKey format:
 *  [5 bytes RID][1 byte Index][1 byte Hash Algorithm][1 byte Encryption Algorithm][20 bytes HashValue][4 bytes Public Key Exponent][2 bytes Modulus Length][Variable bytes Modulus]
 *  Where:
 *   - Hash Algorithm: The only algorithm supported is SHA-1.The value is set to 0x01
 *   - Encryption Algorithm: The encryption algorithm in which this key is used. Currently support only one type: RSA. The value is set to 0x01.
 *   - HashValue: Which is calculated using SHA-1 over the following fields: RID & Index & Modulus & Exponent
 *   - Public Key Exponent: Actually, the real length of the exponent is either one byte or 3 bytes. It can have two values: 3 (Format is 0x00 00 00 03), or  65537 (Format is 0x00 01 00 01)
 *   - Modulus Length: LenL LenH Indicated the length of the next field.
 *   - Modulus: This is the modulus field of the public key. Its length is specified in the field above.
 * @param capkLen the length of capk data buffer
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_setCAPK(IN BYTE * capk,
  IN int              capkLen);

/**
 * Remove Certificate Authority Public Key
 *
 * Removes the CAPK as specified by the RID/Index
 *
 * @param capk 6 byte CAPK =  5 bytes RID + 1 byte INDEX
 * @param capkLen the length of capk data buffer
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_removeCAPK(IN BYTE * capk,
  IN int                 capkLen);

/**
 * Remove All Certificate Authority Public Key
 *
 * Removes all the CAPK
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
emv_removeAllCAPK();

/**
 * Retrieve the Certificate Authority Public Key list
 *
 * Returns all the CAPK RID and Index installed on the terminal.
 *
 * @param keys [key1][key2]...[keyn], each key 6 bytes where
 *  key = 5 bytes RID + 1 byte index
 * @param keysLen the length of keys data buffer
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_retrieveCAPKList(OUT BYTE * keys,
  IN_OUT int *                  keysLen);

/**
 * Retrieve the Certificate Revocation List
 *
 * Returns the CRL entries on the terminal.
 *
 * @param list [CRL1][CRL2]...[CRLn], each CRL 9 bytes where
 *  CRL = 5 bytes RID + 1 byte index + 3 bytes serial number
 * @param lssLen the length of list data buffer
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString()
 */
int
emv_retrieveCRL(OUT BYTE * list,
  IN_OUT int *             lssLen);

/**
 * Set Certificate Revocation List
 *
 * Sets the CRL
 *
 * @param list CRL Entries containing the RID, Index, and serial numbers to set
 *  [CRL1][CRL2]...[CRLn] where each [CRL]  is 9 bytes:
 *  [5 bytes RID][1 byte CAPK Index][3 bytes serial number]
 * @param lsLen the length of list data buffer
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString()
 */
int
emv_setCRL(IN BYTE * list,
  IN int             lsLen);

/**
 * Retrieve the Certificate Revocation List
 *
 * Returns the CRL entries on the terminal.
 *
 * @param list [CRL1][CRL2]...[CRLn], each CRL 9 bytes where
 *  CRL = 5 bytes RID + 1 byte index + 3 bytes serial number
 * @param lssLen the length of list data buffer
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString()
 */
int
emv_removeCRL(IN BYTE * list,
  IN int                lsLen);

/**
 * Remove All Certificate Revocation List Entries
 *
 * Removes all CRLEntry entries
 * @return RETURN_CODE: Values can be parsed with device_getResponseCodeString()
 *
 */
int
emv_removeAllCRL();

/**
 * Start EMV Transaction Request
 *
 * Authorizes the EMV transaction for an ICC card
 *
 * The tags will be returned in the callback routine.
 *
 * @param amount Transaction amount value (tag value 9F02) - SEE
 * IMPORTANT NOTE BELOW
 *
 * @param amtOther Other amount value, if any (tag value 9F03) - SEE
 * IMPORTANT NOTE BELOW
 * @param exponent Number of characters after decimal point
 * @param type Transaction type (tag value 9C).
 * @param timeout Timeout value in seconds.
 * @param tags Any other tags to be included in the request.  Passed
 * as a TLV stream.  Example, tag 9F02 with amount 0x000000000100 would
 * be 0x9F0206000000000100 If tags 9F02 (amount),9F03 (other amount), or
 * 9C (transaction type) are included, they will take priority over
 * these values supplied as individual parameters to this method.
 * @param tagsLen Length of tags
 * @param forceOnline TRUE = do not allow offline approval,  FALSE = allow
 * ICC to approve offline if terminal capable
 *
 * Note: To request tags to be included in default response, use tag
 * DFEE1A, and specify tag list.  Example four tags 9F02, 9F36, 95, 9F37
 * to be included in response = DFEE1A079F029F36959F37
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString
 *>>>>>IMPORTANT: parameters for amount and amtOther MUST BE PASSED AS
 *     A DOUBLE VALUE WITH DECIMAL POINT.  Example, do not pass 1, but
 *     instead pass 1.0 or 1.00. Otherwise, results will be
 *     unpredictable
 */
int
emv_startTransaction(float amount, float amtOther, int exponent, int type, int timeout, BYTE * tags, int tagsLen,
  int forceOnline);

/**
 * Authenticate EMV Transaction Request
 *
 * Authenticates the EMV transaction for an ICC card.  Execute this after
 * receiving response with result code 0x10 to emv_startTransaction
 *
 * The tags will be returned in the callback routine.
 *
 * @param updatedTLV  TLV stream that can be used to update the following values:
 *   - 9F02: Amount
 *   - 9F03: Other amount
 *   - 9C: Transaction type
 *   - 5F57: Account type
 *
 *  In addition tag DFEE1A can be sent to specify tag list to include
 *  in results. Example four tags 9F02, 9F36, 95, 9F37 to be included
 *  in response = DFEE1A079F029F36959F37
 * @param updatedTLVLen
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString
 */
int
emv_authenticateTransaction(BYTE * updatedTLV, int updatedTLVLen);

/**
 * Authenticate EMV Transaction Request with Timeout
 *
 * Authenticates the EMV transaction for an ICC card.  Execute this after
 * receiving response with result code 0x10 to emv_startTransaction
 *
 * The tags will be returned in the callback routine.
 * @param timeout Timeout value in seconds.
 * @param updatedTLV  TLV stream that can be used to update the following values:
 *   - 9F02: Amount
 *   - 9F03: Other amount
 *   - 9C: Transaction type
 *   - 5F57: Account type
 *
 *  In addition tag DFEE1A can be sent to specify tag list to include
 *  in results. Example four tags 9F02, 9F36, 95, 9F37 to be included
 *  in response = DFEE1A079F029F36959F37
 * @param updatedTLVLen
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString
 */
int
emv_authenticateTransactionWithTimeout(int timeout, BYTE * updatedTLV, int updatedTLVLen);

/**
 * Complete EMV Transaction Request
 *
 * Completes the EMV transaction for an ICC card when online
 * authorization request is received from emv_authenticateTransaction
 *
 * The tags will be returned in the callback routine.
 *
 * @param commError Communication error with host.  Set to TRUE(1) if
 * host was unreachable, or FALSE(0) if host response received.  If
 * Communication error, authCode, iad, tlvScripts can be null.
 * @param authCode Authorization code from host.  Two bytes.  Example
 * 0x3030.  (Tag value 8A).  Required
 * @param authCodeLen the length of authCode
 * @param iad Issuer Authentication Data, if any.  Example
 * 0x11223344556677883030 (tag value 91).
 * @param iadLen the length of iadLen
 * @param tlvScripts 71/72 scripts, if any
 * @param tlvScriptsLen the length of tlvScriptsLen
 * @param tlv  Additional TLV data to return with transaction results (if any)
 * @param tlvLen the length of tlv
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString
 *
 *
 * NOTE:  There are three possible outcomes for Authorization Code:  Approval, Refer To Bank, Decline.  The kernel maps these three outcomes to
 * valid authorization response codes using tag DFEE1B through 8 bytes:  {2 bytes Approval Code}{2 bytes Referral Code}{2 bytes Decline Code}{2 bytes RFU}
 * If your gateway uses "00" for Approval, "01" for Referral, and "05" for Decline, then DFEE1B 08 3030 3031 3035 0000
 * If you use an authorization code value that that is not defined in DFEE1B, the kernel will use the DECLINE value of DFEE1B by default.
 */
int
emv_completeTransaction(int commError, BYTE * authCode, int authCodeLen, BYTE * iad, int iadLen, BYTE * tlvScripts,
  int tlvScriptsLen, BYTE * tlv, int tlvLen);

/**
 * Cancel EMV Transaction
 *
 * Cancels the currently executing EMV transaction.
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString()
 */
int
emv_cancelTransaction();

/**
 * Retrieve Transaction Results
 *
 * Retrieves specified EMV tags from the currently executing transaction.
 *
 * @param tags Tags to be retrieved.  Example 0x9F028A will retrieve
 * tags 9F02 and 8A
 * @param tagsLen Length of tag list
 * @param cardData All requested tags returned as unencrypted, encrypted
 * and masked TLV data in IDTTransactionData object
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString()
 */
int
emv_retrieveTransactionResult(IN BYTE * tags,
  IN int                                tagsLen,
  OUT IDTTransactionData *              cardData);

/**
 * Callback Response LCD Display
 *
 * Provides menu selection responses to the kernel after a callback was received with DeviceState.EMVCallback, and
 * callbackType = EMV_CALLBACK_TYPE.EMV_CALLBACK_TYPE_LCD, and lcd_displayMode = EMV_LCD_DISPLAY_MODE_MENU, EMV_LCD_DISPLAY_MODE_PROMPT,
 * or EMV_LCD_DISPLAY_MODE_LANGUAGE_SELECT
 *
 * @param type If Cancel key pressed during menu selection, then value is EMV_LCD_DISPLAY_MODE_CANCEL.  Otherwise, value can be EMV_LCD_DISPLAY_MODE_MENU, EMV_LCD_DISPLAY_MODE_PROMPT,
 * or EMV_LCD_DISPLAY_MODE_LANGUAGE_SELECT
 * @param selection If type = EMV_LCD_DISPLAY_MODE_MENU or EMV_LCD_DISPLAY_MODE_LANGUAGE_SELECT, provide the selection ID line number. Otherwise, if type = EMV_LCD_DISPLAY_MODE_PROMPT
 * supply either 0x43 ('C') for Cancel, or 0x45 ('E') for Enter/accept
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
emv_callbackResponseLCD(int type, byte selection);

/**
 * Callback Response MSR Entry
 *
 * Provides MSR information to kernel after a callback was received with DeviceState.EMVCallback, and
 *  callbackType = EMV_CALLBACK_MSR
 *
 * @param MSR Swiped track data
 * @param MSRLen Swiped track data length
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_callbackResponseMSR(BYTE * MSR, int MSRLen);

/**
 * Retrieve Application Data by AID
 *
 * Retrieves the Application Data as specified by the AID name passed as
 * a parameter.
 *
 * @param AID Name of ApplicationID. Must be between 5 and 16 bytes
 * @param AIDLen the length of AID data buffer.
 * @param tlv  The TLV elements of the requested AID
 * @param tlvLen the length of tlv data buffer.
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
emv_retrieveApplicationData(IN BYTE * AID,
  IN int                              AIDLen,
  OUT BYTE *                          tlv,
  IN_OUT int *                        tlvLen);

/**
 * To register the emv callback function to get the EMV processing response.
 */
void
emv_registerCallBk(pEMV_callBack pEMVf);

/**
 * Allow fallback for EMV transactions.  Default is TRUE
 * @param allow TRUE = allow fallback, FALSE = don't allow fallback
 */
void
emv_allowFallback(IN int allow);

/**
 * Enables authenticate for EMV transactions.  If a emv_startTranaction
 *        results in code 0x0010 (start transaction success), then
 *        emv_authenticateTransaction can automatically execute if
 *        parameter is set to TRUE
 * @param authenticate TRUE = auto authenticate, FALSE = manually authenticate
 */
void
emv_setAutoAuthenticateTransaction(IN int authenticate);

/**
 * Set EMV Transaction Parameters
 *
 * Set the parameters to be used on EMV transactions for an ICC card
 * when Auto Poll is on
 *
 * The tags will be returned in the callback routine.
 *
 * @param amount Transaction amount value  (tag value 9F02)
 * @param amtOther Other amount value, if any  (tag value 9F03)
 * @param type Transaction type (tag value 9C).
 * @param timeout Timeout value in seconds.
 * @param tags Any other tags to be included in the request (Maximum
 *   Length = 500 bytes).  Passed as a string.  Example, tag 9F02 with
 *   amount 0x000000000100 would be "9F0206000000000100" If tags 9F02
 *   (amount),9F03 (other amount), or 9C (transaction type) are
 *   included, they will take priority over these values supplied as
 *   individual parameters to this method.  Note: To request tags to
 *   be included in default response, use tag DFEE1A, and specify tag
 *   list.  Example four tags 9F02, 9F36, 95, 9F37 to be included in
 *   response = DFEE1A079F029F36959F37
 * @param tagsLen the length of tags
 */
void
emv_setTransactionParameters(float amount, float amtOther, int type, int timeout, BYTE * tags, int tagsLen);

/**
 * Enable Logging.
 * @param 1 = ON, 0 = OFF
 */
void
enableLogging(int value);

void
registerAdfLogCallBk(pSendDataLog pFSend, pReadDataLog pFRead);
void
registerLogCallBk(pSendDataLog pFSend, pReadDataLog pFRead);


/**
 * Start Transaction Request
 *
 * Authorizes the transaction for an MSR/CTLS/ICC card
 *
 * The tags will be returned in the callback routine.
 *
 * @param amount Transaction amount value (tag value 9F02)
 *      - SEE IMPORTANT NOTE BELOW
 * @param amtOther Other amount value, if any (tag value 9F03)
 *      - SEE IMPORTANT NOTE BELOW
 * @param type Transaction type (tag value 9C).
 * @param timeout Timeout value in seconds.
 * @param tags Any other tags to be included in the request.  Passed as TLV.
 * Example, tag 9F02 with amount 0x000000000100 would be 0x9F0206000000000100
 * If tags 9F02 (amount),9F03 (other amount), or 9C (transaction type) are
 * included, they will take priority over these values supplied as individual
 * parameters to this method.
 *
 * @param tagsLen The length of tags data buffer.
 *
 * >>>>>IMPORTANT: parameters for amount and amtOther MUST BE PASSED AS
 *    A DOUBLE VALUE WITH DECIMAL POINT.  Example, do not pass 1, but
 *    instead pass 1.0 or 1.00. Otherwise, results will be
 *    unpredictable
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 * Note: if auto poll is on, it will returm the error
 * IDG_P2_STATUS_CODE_COMMAND_NOT_ALLOWED
 *
 * NOTE ON APPLEPAY VAS:
 * To enable ApplePay VAS, first a merchant record must be defined in
 * one of the six available index positions (1-6) using
 * device_setMerchantRecord, then container tag FFEE06 must be sent as
 * part of the additional tags parameter of device_startTransaction.
 * Tag FFEE06 must contain tag 9F26 and 9F22, and can optionanally
 * contain tags 9F2B and DFO1.  Example
 * FFEE06189F220201009F2604000000009F2B050100000000DF010101
 *
 * 9F22 = two bytes = ApplePay Terminal Applicaiton Version Number.
 *        Hard defined as 0100 for now. (required)
 * 9F26 = four bytes = ApplePay Terminal Capabilities Information (required)
 *  - Byte 1 = RFU
 *  - Byte 2 = Terminal Type
 *  - - Bit 8 = VAS Support     (1=on, 0 = off)
 *  - - Bit 7 = Touch ID Required  (1=on, 0 = off)
 *  - - Bit 6 = RFU
 *  - - Bit 5 = RFU
 *  - - Bit 1,2,3,4
 *  - - - 0 = Payment Terminal
 *  - - - 1 = Transit Terminal
 *  - - - 2 = Access Terminal
 *  - - - 3 = Wireless Handoff Terminal
 *  - - - 4 = App Handoff Terminal
 *  - - - 15 = Other Terminal
 *  - Byte 3 = RFU
 *  - Byte 4 = Terminal Mode
 *  - - 0 = ApplePay VAS OR ApplePay
 *  - - 1 = ApplePay VAS AND ApplePay
 *  - - 2 = ApplePay VAS ONLY
 *  - - 3 = ApplePay ONLY
 *  9F2B = 5 bytes = ApplePay VAS Filter.  Each byte filters for that
 *         specific merchant index  (optional)
 *  DF01 = 1 byte = ApplePay VAS Protocol.  (optional)
 *  - - Bit 1 : 1 = URL VAS, 0 = Full VAS
 *  - - Bit 2 : 1 = VAS Beeps, 0 = No VAS Beeps
 *  - - Bit 3 : 1 = Silent Comm Error, 2 = EMEA Comm Error
 *  - - Bit 4-8 : RFU
 *
 */
int
device_startTransaction(float amount, float amtOther, int type, const int _timeout, BYTE * tags, int tagsLen);

/**
 * Cancel Transaction request.
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_cancelTransaction();

/**
 * Set Cancel Transaction Mode
 *
 * Set the cancel transaction mode to be with or without LCD message
 *
 *
 * @param mode 0: With LCD message 1: Without LCD message
 *
 */
int
device_setCancelTransactionMode(int mode);

/**
 * Cancel Transaction Silent
 *
 * Cancel transaction with or without showing the LCD message
 *
 *
 * @param enable 0: With LCD message 1: Without LCD message
 * @return success or error code.  Values can be parsed with device_getIDGStatusCodeString
 *
 */
int
device_cancelTransactionSilent(int enable);

/**
 * Polls device for Firmware Version
 *
 * @param firmwareVersion Response returned of Firmware Version
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
device_getFirmwareVersion(char * firmwareVersion);

/**
 * Polls device for Firmware Version
 *
 * @param firmwareVersion Response returned of Firmware Version
 * @param firmwareVersionLen Length of Firmware Version
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
device_getFirmwareVersion_Len(OUT char * firmwareVersion, IN_OUT int * firmwareVersionLen);

/**
 * Send a Command to NEO device
 *
 * Sends a command  to the NEO device .
 *
 * @param cmd  command to execute.
 * @param subCmd, sub command to execute.
 *
 * @param data buffer of NEO command data.
 * @param dataLen, the length of the buffer data.
 *
 * @param response Response data
 * @param respLen, the length of Response data
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_SendDataCommandNEO(int cmd, int subCmd, BYTE * data, int dataLen, BYTE * response, int * respLen);

/**
 * Send Vivo Command Protocol 2 Extended
 *
 * Sends a protocol 2 command to Vivo readers (IDG/NEO)
 *
 *   @param command Command
 *   @param subCommand Sub-Command
 *   @param data Data.  May be null
 *   @param dataLen Data length
 *   @param response Response
 *   @param respLen Response length
 *   @param timeout Timeout, in milliseconds (3000 = 3 seconds)
 *   @param noResponse  TRUE = don't wait for response, FALSE = wait for response defined by timeout
 *   @return success or error code.  Values can be parsed with device_getResponseCodeString
 *   @see ErrorCode
 */
int
device_sendVivoCommandP2_ext(BYTE cmd, BYTE subCmd, BYTE * data, int dataLen, BYTE * response, int * respLen,
  int timeout, int noResponse);

/**
 * Send Vivo Command Protocol 3 Extended
 *
 * Sends a protocol 3 command to Vivo readers (IDG/NEO)
 *
 *   @param command Command
 *   @param subCommand Sub-Command
 *   @param data Data.  May be null
 *   @param dataLen Data length
 *   @param response Response
 *   @param respLen Response length
 *   @param timeout Timeout, in milliseconds (3000 = 3 seconds)
 *   @param noResponse  TRUE = don't wait for response, FALSE = wait for response defined by timeout
 *   @return success or error code.  Values can be parsed with device_getResponseCodeString
 *   @see ErrorCode
 */
int
device_sendVivoCommandP3_ext(BYTE cmd, BYTE subCmd, BYTE * data, int dataLen, BYTE * response, int * respLen,
  int timeout, int noResponse);

/**
 * Update Firmware
 * Updates the firmware of NEO 2 devices.
 * @param firmwareData Signed binary data of a firmware file provided by IDTech
 * @param firmwareDataLen Length of firmwareData
 * @param firmwareName Firmware name.
 *  - For example "VP5300_v1.00.023.0167.S_Test.fm"
 * @param encryptionType Encryption type
 * - 0 : Plaintext
 * - 1 : TDES ECB, PKCS#5 padding
 * - 2 : TDES CBC, PKCS#5, IV is all 0
 * @param keyBlob Encrypted firmware session key blob, TR-31 Rev B, wrapped by
 *  FW Key (Optional, none if firmware is plaintext)
 * @param keyBlobLen Length of keyBlob
 * @return RETURN_CODE:  Values can be parsed with errorCode.getErrorString()
 *
 * Firmware update status is returned in the callback with the following values:
 * sender = device type
 * state = DEVICE_FIRMWARE_UPDATE
 * current block
 * total blocks
 * ResultCode:
 * - RETURN_CODE_DO_SUCCESS = Firmware Update Completed Successfully
 * - RETURN_CODE_BLOCK_TRANSFER_SUCCESS = Current block transferred successfully
 * - Any other return code represents an error condition
 *
 */
int
device_updateFirmware(BYTE * firmwareData, int firmwareDataLen, char * firmwareName, int encryptionType, BYTE * keyBlob,
  int keyBlobLen);

/**
 * Enables pass through mode for ICC. Required when direct ICC commands are required
 * (power on/off ICC, exchange APDU)
 *
 * @return success or error code.  Values can be parsed with device_getResponseCodeString
 * @see ErrorCode
 */
int
device_passthroughOnICC();

/**
 * Disables pass through mode for ICC. Required when executing transactions
 * (start EMV, start MSR, authenticate transaction)
 *
 * @return success or error code.
 * @see ErrorCode
 */
int
device_passthroughOffICC();

/**
 * Enable Pass Through
 *
 * Enables Pass Through Mode for direct communication with L1 interface (power on icc, send apdu, etc).
 *
 * @param enablePassThrough 1 = pass through ON, 0 = pass through OFF
 *
 *
 * @return RETURN_CODE:	Values can be parsed with errorCode.getErrorString()
 */
int
device_enablePassThrough(int enablePassThrough);

/**
 * Send Burst Mode
 *
 * Sets the burst mode for the device.
 *
 * @param mode 0 = OFF, 1 = Always On, 2 = Auto Exit
 *
 * @return success or error code.  Values can be parsed with device_getResponseCodeString
 * @see ErrorCode
 */
int
device_setBurstMode(BYTE mode);

/**
 * Set Poll Mode
 *
 * Sets the poll mode forthe device. Auto Poll keeps reader active, Poll On Demand only polls when requested by terminal
 *
 * @param mode 0 = Auto Poll, 1 = Poll On Demand
 *
 * @return RETURN_CODE:  Values can be parsed with errorCode.getErrorString()
 */
int
device_setPollMode(BYTE mode);

/**
 * Poll for Token
 *
 * Polls for a PICC
 *
 * @param timeout timeout in milliseconds, must be multiple of 10 milliseconds. 30, 120, 630, or 1150 for example.
 *
 * @param respData Response data will be stored in respData.  1 byte of card type, and the Serial Number (or the UID) of the PICC if available.
 * @param respDataLen Length of systemCode.
 *
 * @return RETURN_CODE: Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_pollForToken(IN int timeout, OUT BYTE * respData, IN_OUT int * respDataLen);

/**
 * Set Auto Poll Thread
 * INTERNAL USE ONLY
 */
void
startAutoPollThread();

/**
 * Set Merchant Record
 * Sets the merchant record for ApplePay VAS
 *
 * @param index Merchant Record index, valid values 1-6
 * @param enabled Merchant Enabled/Valid flag
 * @param merchantID  Merchant unique identifer registered with Apple.  Example com.idtechproducts.applePay
 * @param merchantURL Merchant URL, when applicable
 *
 * @return RETURN_CODE:	Values can be parsed with device_getIDGStatusCodeString()
 *
 */

int
device_setMerchantRecord(int index, int enabled, char * merchantID, char * merchantURL);

/**
 * Get Merchant Record
 *
 * Gets the merchant record for the device.
 *
 * @param index Merchant Record index, valid values 1-6
 * @param record
 *            response data from reader.
 *            Merchant Record Index: 1 byte
 *            enabled: 1 byte
 *            Merchant ID: 32 bytes
 *            Length of Merchant URL: 1 byte
 *            Merchant URL: 64 bytes
 *
 * @return success or error code.  Values can be parsed with device_getIDGStatusCodeString()
 * @see ErrorCode
 */
int
device_getMerchantRecord(IN int index, OUT BYTE * record);

/**
 * Get Transaction Results
 * Gets the transaction results when the reader is functioning in "Auto Poll" mode
 *
 * @param cardData The transaction results
 *
 * @return success or error code.  Values can be parsed with device_getResponseCodeString
 * @see ErrorCode
 */
int
device_getTransactionResults(IDTMSRData * cardData);

/**
 * Ping Device
 *
 *
 * Pings the reader.  If connected, returns success.  Otherwise, returns timeout.
 *
 * @return RETURN_CODE:	Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_pingDevice();

/**
 * Control User Interface
 *
 *
 * Controls the User Interface:  Display, Beep, LED
 *
 *  @param values Four bytes to control the user interface
 *  Byte[0] = LCD Message
 *  Messages 00-07 are normally controlled by the reader.
 *  - 00h: Idle Message (Welcome)
 *  - 01h: Present card (Please Present Card)
 *  - 02h: Time Out or Transaction cancel (No Card)
 *  - 03h: Transaction between reader and card is in the middle (Processing...)
 *  - 04h: Transaction Pass (Thank You)
 *  - 05h: Transaction Fail (Fail)
 *  - 06h: Amount (Amount $ 0.00 Tap Card)
 *  - 07h: Balance or Offline Available funds (Balance $ 0.00) Messages 08-0B are controlled by the terminal
 *  - 08h: Insert or Swipe card (Use Chip & PIN)
 *  - 09h: Try Again(Tap Again)
 *  - 0Ah: Tells the customer to present only one card (Present 1 card only)
 *  - 0Bh: Tells the customer to wait for authentication/authorization (Wait)
 *  - FFh: indicates the command is setting the LED/Buzzer only.
 *  Byte[1] = Beep Indicator
 *  - 00h: No beep
 *  - 01h: Single beep
 *  - 02h: Double beep
 *  - 03h: Three short beeps
 *  - 04h: Four short beeps
 *  - 05h: One long beep of 200 ms
 *  - 06h: One long beep of 400 ms
 *  - 07h: One long beep of 600 ms
 *  - 08h: One long beep of 800 ms
 *  Byte[2] = LED Number
 *  - 00h: LED 0 (Power LED) 01h: LED 1
 *  - 02h: LED 2
 *  - 03h: LED 3
 *  - FFh: All LEDs
 *  Byte[3] = LED Status
 *  - 00h: LED Off
 *  - 01h: LED On
 * @return RETURN_CODE:	Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_controlUserInterface(IN BYTE * values);

/**
 * Control Indicators
 *
 *
 * Control the reader.  If connected, returns success.  Otherwise, returns timeout.
 *
 *    @param indicator description as follows:
 *    - 00h: ICC LED
 *    - 01h: Blue MSR
 *    - 02h: Red MSR
 *    - 03h: Green MSR
 *    @param enable  TRUE = ON, FALSE = OFF
 *    @return success or error code.  Values can be parsed with device_getResponseCodeString
 *    @see ErrorCode
 */
int
device_controlIndicator(IN int indicator, IN int enable);

/**
 * set RTC date and time of the device
 *
 * @param dateTime
 *            <dateTime Data> is:
 *                 6 byte data for YYMMDDHHMMSS in hex.  For example 0x171003102547 stands for 2017 Oct 3rd 10:25:47
 *
 * @param dateTimeLen
 *                 should be always 6 bytes
 *
 * @return success or error code. Values can be parsed with device_getResponseCodeString
 * @see ErrorCode
 */
int
device_setRTCDateTime(BYTE * dateTime, int dateTimeLen);

/**
 * get RTC date and time of the device
 *
 * @param dateTime
 *            <dateTime Data> is:
 *                 6 byte data for YYMMDDHHMMSS in hex.  For example 0x171003102547 stands for 2017 Oct 3rd 10:25:47
 *
 * @param dateTimeLen
 *                 should be always 6 bytes
 *
 * @return success or error code. Values can be parsed with device_getResponseCodeString
 * @see ErrorCode
 */
int
device_getRTCDateTime(BYTE * dateTime, int * dateTimeLen);

/**
 * Power On ICC
 *
 * Power up the currently selected microprocessor card in the ICC reader
 *@param ATR, the ATR data response when succeeded power on ICC,
 *@param inLen, the length of ATR data
 *
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString()
 */
int
icc_powerOnICC(OUT BYTE * ATR, IN_OUT int * inLen);

/**
 * Power Off ICC
 *
 * Powers down the ICC
 *
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString()
 *
 * If Success, empty
 * If Failure, ASCII encoded data of error string
 */
int
icc_powerOffICC();

/**
 * Exchange APDU with plain text
 * For Non-SRED Augusta Only
 *
 * Sends an APDU packet to the ICC.  If successful, response is the APDU data in response parameter.
 *
 * @param c_APDU	APDU data packet
 * @param cLen  APDU data packet length
 * @param reData    Unencrypted APDU response
 * @param reLen      Unencrypted APDU response data length
 *
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString
 */
int
icc_exchangeAPDU(IN BYTE * c_APDU, IN int cLen, OUT BYTE * reData, IN_OUT int * reLen);

/**
 * Get Reader Status
 *
 * Returns the reader status
 *
 * @param status Pointer that will return with the ICCReaderStatus results.
 *	bit 0:  0 = ICC Power Not Ready, 1 = ICC Powered
 *	bit 1:  0 = Card not seated, 1 = card seated
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString
 */
int
icc_getICCReaderStatus(OUT BYTE * status);


/**
 * To register the msr callback function to get the MSR card data.
 */
void
msr_registerCallBk(pMSR_callBack pMSRf);

/**
 * To register the msr callback function to get the MSR card data pointer.
 */
void
msr_registerCallBkp(pMSR_callBackp pMSRf);

/**
 * Start MSR Swipe
 *
 * Enables MSR, waiting for swipe to occur. Allows track
 * selection. Returns IDTMSRData instance to swipeMSRData()
 * @param timeout Swipe Timeout Value
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 * Note: if auto poll mode is on, it will return command not allowed error
 */
int
msr_startMSRSwipe(IN int _timeout);
int
msr_startMSRSwipe(int _timeout);

/**
 * Disable MSR Swipe
 * Cancels MSR swipe request.
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
msr_cancelMSRSwipe();

/**
 * Start Transaction Request
 *
 * Authorizes the transaction for an MSR/CTLS/ICC card
 *
 * The tags will be returned in the callback routine.
 *
 * @param amount Transaction amount value	(tag value 9F02) - SEE IMPORTANT NOTE BELOW
 * @param amtOther Other amount value, if any	(tag value 9F03) - SEE IMPORTANT NOTE BELOW
 * @param type Transaction type (tag value 9C).
 * @param timeout Timeout value in seconds.
 * @param tags Any other tags to be included in the request.  Passed as a byte array.	Example, tag 9F0C with amount 0x000000000100 would be 0x9F0C06000000000100
 *  If tags 9F02 (amount),9F03 (other amount), or 9C (transaction type) are included, they will take priority over these values supplied as individual parameters to this method.
 *
 * @param tagsLen The length of tags data buffer.
 * @param interfaces Interfaces to use:
 * bit 0 = MSR
 * bit 1 = Contact
 * bit 2 = CTLS
 * bits 3-7 = RFU.
 *
 *
 * >>>>>IMPORTANT: parameters for amount and amtOther MUST BE PASSED AS A DOUBLE VALUE WITH DECIMAL POINT.  Example, do not pass 1, but instead pass 1.0 or 1.00. Otherwise, results will be unpredictable
 *
 * @return RETURN_CODE:	Values can be parsed with device_getIDGStatusCodeString()
 * Note: if auto poll is on, it will returm the error IDG_P2_STATUS_CODE_COMMAND_NOT_ALLOWED
 *
 * NOTE ON APPLEPAY VAS:
 * To enable ApplePay VAS, first a merchant record must be defined in one of the six available index positions (1-6) using device_setMerchantRecord, then container tag FFEE06
 * must be sent as part of the additional tags parameter of device_startTransaction.  Tag FFEE06 must contain tag 9F26 and 9F22, and can optionanally contain tags 9F2B and DFO1.
 * Example FFEE06189F220201009F2604000000009F2B050100000000DF010101
 * 9F22 = two bytes = ApplePay Terminal Applicaiton Version Number.	Hard defined as 0100 for now. (required)
 * 9F26 = four bytes = ApplePay Terminal Capabilities Information (required)
 *  - Byte 1 = RFU
 *  - Byte 2 = Terminal Type
 *  - - Bit 8 = VAS Support	(1=on, 0 = off)
 *  - - Bit 7 = Touch ID Required  (1=on, 0 = off)
 *  - - Bit 6 = RFU
 *  - - Bit 5 = RFU
 *  - - Bit 1,2,3,4
 *  - - - 0 = Payment Terminal
 *  - - - 1 = Transit Terminal
 *  - - - 2 = Access Terminal
 *  - - - 3 = Wireless Handoff Terminal
 *  - - - 4 = App Handoff Terminal
 *  - - - 15 = Other Terminal
 *  - Byte 3 = RFU
 *  - Byte 4 = Terminal Mode
 *  - - 0 = ApplePay VAS OR ApplePay
 *  - - 1 = ApplePay VAS AND ApplePay
 *  - - 2 = ApplePay VAS ONLY
 *  - - 3 = ApplePay ONLY
 *  9F2B = 5 bytes = ApplePay VAS Filter.  Each byte filters for that specific merchant index  (optional)
 *  DF01 = 1 byte = ApplePay VAS Protocol.  (optional)
 *  - - Bit 1 : 1 = URL VAS, 0 = Full VAS
 *  - - Bit 2 : 1 = VAS Beeps, 0 = No VAS Beeps
 *  - - Bit 3 : 1 = Silent Comm Error, 2 = EMEA Comm Error
 *  - - Bit 4-8 : RFU
 *
 */
int
L2_startTransaction(float amount, float amtOther, int type, const int _timeout, BYTE * tags, int tagsLen,
  byte interfaces);

/**
 * Continue EMV Transaction after reviewing captured card tags
 *
 * Continue the EMV transaction for an ICC card.  Execute this after receiving response with result code 0x10 to emv_startTransaction
 *
 * The tags will be returned in the callback routine.
 *
 * @param updatedTLV  TLV stream that can be used to update the following values:
 *   - 9F02: Amount
 *   - 9F03: Other amount
 *   - 9C: Transaction type
 *   - 5F57: Account type
 *  In addition tag DFEE1A can be sent to specify tag list to include in results. Example four tags 9F02, 9F36, 95, 9F37 to be included in response = DFEE1A079F029F36959F37
 * @param updatedTLVLen
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString
 */
int
L2_continueTransaction(BYTE * updatedTLV, int updatedTLVLen);

/**
 * Complete EMV Transaction Request
 *
 *
 * Completes the EMV transaction for an ICC card when online authorization request is received from emv_authenticateTransaction
 *
 * The tags will be returned in the callback routine.
 *
 * @param commError Communication error with host.  Set to TRUE(1) if host was unreachable, or FALSE(0) if host response received.  If Communication error, authCode, iad, tlvScripts can be null.
 * @param authCode Authorization code from host.  Two bytes.  Example 0x3030.  (Tag value 8A).  Required
 * @param authCodeLen the length of authCode
 * @param iad Issuer Authentication Data, if any.  Example 0x11223344556677883030 (tag value 91).
 * @param iadLen the length of iadLen
 * @param tlvScripts 71/72 scripts, if any
 * @param tlvScriptsLen the length of tlvScriptsLen
 * @param tlv  Additional TVL data to return with transaction results (if any)
 * @param tlvLen the length of tlv
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString
 */
int
L2_completeTransaction(int commError, BYTE * authCode, int authCodeLen,
  BYTE * iad, int iadLen, BYTE * tlvScripts, int tlvScriptsLen, BYTE * tlv, int tlvLen);

/**
 * Cancel Transaction
 *
 * Cancels the currently executing transaction.
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString()
 */
int
L2_cancelTransaction();

/**
 * To register the emv callback function to get the EMV processing response.
 */
void
L2_registerEMVCallBk(pEMV_callBack pEMVf);

/**
 * To register the callback function to get the MSR fallbaack processing response or the CTLS MSD processing response.
 */
void
L2_registerMSRCallBk(pMSR_callBack pMSRf);

/**
 * To register the callback function to get the MSR fallbaack processing response or the CTLS MSD processing response. The callback parameter uses pointer instead structure
 */
void
L2_registerMSRCallBkp(pMSR_callBackp pEMVf);

/**
 * Callback Response to Menu Selection Request
 *
 * Provides menu selection responses to the kernel after a callback was received with DeviceState.EMVCallback, and
 * callbackType = EMV_CALLBACK_TYPE.EMV_CALLBACK_TYPE_LCD, and lcd_displayMode = EMV_LCD_DISPLAY_MODE_MENU, EMV_LCD_DISPLAY_MODE_PROMPT,
 * or EMV_LCD_DISPLAY_MODE_LANGUAGE_SELECT
 *
 * @param type If Cancel key pressed during menu selection, then value is EMV_LCD_DISPLAY_MODE_CANCEL.  Otherwise, value can be EMV_LCD_DISPLAY_MODE_MENU, EMV_LCD_DISPLAY_MODE_PROMPT,
 * or EMV_LCD_DISPLAY_MODE_LANGUAGE_SELECT
 * @param selection If type = EMV_LCD_DISPLAY_MODE_MENU or EMV_LCD_DISPLAY_MODE_LANGUAGE_SELECT, provide the selection ID line number. Otherwise, if type = EMV_LCD_DISPLAY_MODE_PROMPT
 * supply either 0x43 ('C') for Cancel, or 0x45 ('E') for Enter/accept
 *
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString()
 *
 */
int
L2_callbackResponseMenu(IN int type, byte selection);

/**
 * Retrieve Transaction Results
 *
 *
 * Retrieves specified EMV tags from the currently executing transaction.
 *
 * @param tags Tags to be retrieved.  Example 0x9F028A will retrieve tags 9F02 and 8A
 * @param tagsLen Length of tag list
 * @param cardData All requested tags returned as unencrypted, encrypted and masked TLV data in IDTTransactionData object
 *
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString()
 *
 */
int
L2_retrieveTransactionResult(IN BYTE * tags, IN int tagsLen, IDTTransactionData * cardData);

/**
 * Retrieve CTLS Application Data by AID
 *
 * Retrieves the CTLS Application Data as specified by the AID name passed as a parameter.
 *
 * @param AID Name of ApplicationID. Must be between 5 and 16 bytes
 * @param AIDLen the length of AID data buffer.
 * @param tlv  The TLV elements of the requested AID
 * @param tlvLen the length of tlv data buffer.
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
L2_ctls_retrieveApplicationData(IN BYTE * AID, IN int AIDLen, OUT BYTE * tlv, IN_OUT int * tlvLen);

/**
 * Set CTLS Application Data by AID
 *
 * Sets the Application Data for CTLS as specified by TLV data
 *
 * @param tlv  Application data in TLV format
 *  The first tag of the TLV data must be the group number (FFE4).
 *  The second tag of the TLV data must be the AID (9F06)
 *
 *  Example valid TLV, for Group #2, AID a0000000035010:
 *  "ffe401029f0607a0000000051010ffe10101ffe50110ffe30114ffe20106"
 * @param tlvLen the length of tlv data buffer
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
L2_ctls_setApplicationData(IN BYTE * tlv, IN int tlvLen);

/**
 * Remove CTLS Application Data by AID
 * Removes the Application Data for CTLS as specified by the AID name passed as a parameter
 *
 * @param AID Name of ApplicationID Must be between 5 and 16 bytes
 *
 * @param AIDLen the length of AID data buffer
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
L2_ctls_removeApplicationData(IN BYTE * AID, IN int AIDLen);

/**
 * Removes All CTLS Application Data
 *
 * Removes all the Application Data
 *
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
L2_ctls_removeAllApplicationData();

/**
 * Retrieve All CTLS AIDs groups/names
 *
 * Returns all the AIDs on the terminal for CTLS.  An AID is defined by the group it is in and the AID name
 *
 * @param AIDList  AIDS a repeating sequence,  <1 byte group><1 byte AID len><AID name>. . .
 * @param AIDListLen  the length of AIDList buffer
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
L2_ctls_retrieveAIDList(OUT BYTE * AIDList, IN_OUT int * AIDListLen);

/**
 * Retrieve CTLS Terminal Data
 *
 *
 * Retrieves the Terminal Data for CTLS. This is configuration group 0 (Tag DFEE2D - > DFEE2D0100).  The terminal data
 *  can also be retrieved by L2_ctls_getConfigurationGroup(0).
 *
 * @param tlv Response returned as a TLV
 * @param tlvLen the length of tlv data buffer
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 *
 */
int
L2_ctls_retrieveTerminalData(OUT BYTE * tlv, IN_OUT int * tlvLen);

/**
 * Set CTLS Terminal Data
 *
 * Sets the Terminal Data for CTLS as specified by the TLV.  The first TLV must be Configuration Group Number (Tag DFEE2D).	The terminal global data
 *  is group 0, so the first TLV would be DFEE2D0100.  Other groups can be defined using this method (1 or greater), and those can be
 *  retrieved with ctls_getConfigurationGroup(int group), and deleted with ctls_removeConfigurationGroup(int group).	You cannot
 *  delete group 0.
 *
 * @param tlv TerminalData configuration file
 * @param tlvLen the length of tlv data buffer
 *
 * @retval RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
L2_ctls_setTerminalData(IN BYTE * tlv, IN int tlvLen);

/**
 * Retrieve CTLS Certificate Authority Public Key
 *
 *
 * Retrieves the CAPK for CTLS as specified by the RID/Index	passed as a parameter.
 *
 * @param capk 6 bytes CAPK = 5 bytes RID + 1 byte Index
 * @param capkLen the length of capk data buffer
 * @param key Response returned as a CAKey format:
 *  [5 bytes RID][1 byte Index][1 byte Hash Algorithm][1 byte Encryption Algorithm]
 *  [20 bytes HashValue][4 bytes Public Key Exponent][2 bytes Modulus Length][Variable bytes Modulus]
 *  Where:
 *   - Hash Algorithm: The only algorithm supported is SHA-1.The value is set to 0x01
 *   - Encryption Algorithm: The encryption algorithm in which this key is used. Currently support only one type: RSA. The value is set to 0x01.
 *   - HashValue: Which is calculated using SHA-1 over the following fields: RID & Index & Modulus & Exponent
 *   - Public Key Exponent: Actually, the real length of the exponent is either one byte or 3 bytes. It can have two values: 3 (Format is 0x00 00 00 03), or  65537 (Format is 0x00 01 00 01)
 *   - Modulus Length: LenL LenH Indicated the length of the next field.
 *   - Modulus: This is the modulus field of the public key. Its length is specified in the field above.
 * @param keyLen the length of key data buffer
 * * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
L2_ctls_retrieveCAPK(IN BYTE * capk, IN int capkLen, OUT BYTE * key, IN_OUT int * keyLen);

/**
 * Set CTLSCertificate Authority Public Key
 *
 * Sets the CAPK as specified by the CAKey structure
 *
 * @param capk CAKey format:
 *  [5 bytes RID][1 byte Index][1 byte Hash Algorithm][1 byte Encryption Algorithm][20 bytes HashValue][4 bytes Public Key Exponent][2 bytes Modulus Length][Variable bytes Modulus]
 *  Where:
 *   - Hash Algorithm: The only algorithm supported is SHA-1.The value is set to 0x01
 *   - Encryption Algorithm: The encryption algorithm in which this key is used. Currently support only one type: RSA. The value is set to 0x01.
 *   - HashValue: Which is calculated using SHA-1 over the following fields: RID & Index & Modulus & Exponent
 *   - Public Key Exponent: Actually, the real length of the exponent is either one byte or 3 bytes. It can have two values: 3 (Format is 0x00 00 00 03), or  65537 (Format is 0x00 01 00 01)
 *   - Modulus Length: LenL LenH Indicated the length of the next field.
 *   - Modulus: This is the modulus field of the public key. Its length is specified in the field above.
 * @param capkLen the length of capk data buffer
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
L2_ctls_setCAPK(IN BYTE * capk, IN int capkLen);

/**
 * Remove CTLS Certificate Authority Public Key
 *
 * Removes the CAPK as specified by the RID/Index
 *
 * @param capk 6 byte CAPK =  5 bytes RID + 1 byte INDEX
 * @param capkLen the length of capk data buffer
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
L2_ctls_removeCAPK(IN BYTE * capk, IN int capkLen);

/**
 * Remove All CTLS Certificate Authority Public Key
 *
 * Removes all the CTLS CAPK
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
L2_ctls_removeAllCAPK();

/**
 * Retrieve the CTLS Certificate Authority Public Key list
 *
 * Returns all the CTLS CAPK RID and Index installed on the terminal.
 *
 * @param keys [key1][key2]...[keyn], each key 6 bytes where
 *  key = 5 bytes RID + 1 byte index
 * @param keysLen the length of keys data buffer
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
L2_ctls_retrieveCAPKList(OUT BYTE * keys, IN_OUT int * keysLen);

/**
 * Set Configuration Group
 *
 * Sets the Configuration Group for CTLS as specified by the TLV data
 *
 * @param tlv  Configuration Group Data in TLV format
 *  The first tag of the TLV data must be the group number (DFEE2D).
 *  A second tag must exist
 * @param tlvLen the length of tlv data buffer
 *
 *
 * @return RETURN_CODE:	Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
L2_setConfigurationGroup(IN BYTE * tlv, IN int tlvLen);

/**
 * Get Configuration Group
 *
 * Retrieves the Configuration for the specified Group.
 *
 * @param group Configuration Group
 * @param tlv return data
 *
 * @param tlvLen the length of tlv data buffer
 * @return RETURN_CODE:	Values can be parsed with device_getIDGStatusCodeString()
 *
 */

int
L2_getConfigurationGroup(IN int group, OUT BYTE * tlv, OUT int * tlvLen);

/**
 * Allow fallback for EMV transactions.  Default is TRUE
 * @param allow TRUE = allow fallback, FALSE = don't allow fallback
 */
void
L2_allowFallback(IN int allow);

/**
 * Retrieve All Configuration Groups
 *
 * Returns all the Configuration Groups installed on the terminal for CTLS
 *
 * @param tlv  The TLV elements data
 * @param tlvLen the length of tlv data buffer.
 *
 * @return RETURN_CODE:	Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
L2_getAllConfigurationGroups(OUT BYTE * tlv, IN_OUT int * tlvLen);

/**
 * Remove Configuration Group
 *
 * Removes the Configuration as specified by the Group.  Must not by group 0
 *
 * @param group Configuration Group
 *
 * @retval RETURN_CODE:	Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
L2_removeConfigurationGroup(int group);

/**
 * Retrieve Application Data by AID
 *
 * Retrieves the Application Data as specified by the AID name passed as a parameter.
 *
 * @param AID Name of ApplicationID. Must be between 5 and 16 bytes
 * @param AIDLen the length of AID data buffer.
 * @param tlv  The TLV elements of the requested AID
 * @param tlvLen the length of tlv data buffer.
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
L2_retrieveApplicationData(IN BYTE * AID, IN int AIDLen, OUT BYTE * tlv, IN_OUT int * tlvLen);

/**
 * Set Application Data by AID
 *
 * Sets the Application Data as specified by the application name and TLV data
 *
 * @param name Application name, 10-32 ASCII hex characters representing 5-16 bytes  Example "a0000000031010"
 * @param nameLen the length of name data buffer of Application name,
 * @param tlv  Application data in TLV format
 * @param tlvLen the length of tlv data buffer
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString()
 */
int
L2_setApplicationData(IN BYTE * name, IN int nameLen, IN BYTE * tlv, IN int tlvLen);

/**
 * Remove Application Data by AID
 * Removes the Application Data for CTLS as specified by the AID name passed as a parameter
 *
 * @param AID Name of ApplicationID Must be between 5 and 16 bytes
 *
 * @param AIDLen the length of AID data buffer
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
L2_removeApplicationData(IN BYTE * AID, IN int AIDLen);

/**
 * Remove All Application Data
 *
 * Removes all the Application Data
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
L2_removeAllApplicationData();

/**
 * Retrieve AID list
 *
 * Returns all the AID names installed on the terminal for CTLS. .
 *
 * @param AIDList  array of AID name byte arrays
 * @param AIDListLen  the length of AIDList array buffer
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
L2_retrieveAIDList(OUT BYTE * AIDList, IN_OUT int * AIDListLen);

/**
 * Retrieve Terminal Data
 *
 *
 * Retrieves the Terminal Data for ICC EMV.
 *
 * @param tlv Response returned as a TLV
 * @param tlvLen the length of tlv data buffer
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
L2_retrieveTerminalData(OUT BYTE * tlv, IN_OUT int * tlvLen);

/**
 * Set Terminal Data
 *
 * Sets the Terminal Data for ICC EMV
 *
 * @param tlv TerminalData TLV terminal data
 * @param tlvLen the length of tlv data buffer
 *
 * @retval RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
L2_setTerminalData(IN BYTE * tlv, IN int tlvLen);

/**
 * Retrieve Certificate Authority Public Key
 *
 *
 * Retrieves the CAPK for ICC EMV as specified by the RID/Index	passed as a parameter.
 *
 * @param capk 6 bytes CAPK = 5 bytes RID + 1 byte Index
 * @param capkLen the length of capk data buffer
 * @param key Response returned as a CAKey format:
 *  [5 bytes RID][1 byte Index][1 byte Hash Algorithm][1 byte Encryption Algorithm]
 *  [20 bytes HashValue][4 bytes Public Key Exponent][2 bytes Modulus Length][Variable bytes Modulus]
 *  Where:
 *   - Hash Algorithm: The only algorithm supported is SHA-1.The value is set to 0x01
 *   - Encryption Algorithm: The encryption algorithm in which this key is used. Currently support only one type: RSA. The value is set to 0x01.
 *   - HashValue: Which is calculated using SHA-1 over the following fields: RID & Index & Modulus & Exponent
 *   - Public Key Exponent: Actually, the real length of the exponent is either one byte or 3 bytes. It can have two values: 3 (Format is 0x00 00 00 03), or  65537 (Format is 0x00 01 00 01)
 *   - Modulus Length: LenL LenH Indicated the length of the next field.
 *   - Modulus: This is the modulus field of the public key. Its length is specified in the field above.
 * @param keyLen the length of key data buffer
 * * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
L2_retrieveCAPK(IN BYTE * capk, IN int capkLen, OUT BYTE * key, IN_OUT int * keyLen);

/**
 * Set Certificate Authority Public Key
 *
 * Sets the CAPK as specified by the CAKey structure
 *
 * @param capk CAKey format:
 *  [5 bytes RID][1 byte Index][1 byte Hash Algorithm][1 byte Encryption Algorithm][20 bytes HashValue][4 bytes Public Key Exponent][2 bytes Modulus Length][Variable bytes Modulus]
 *  Where:
 *   - Hash Algorithm: The only algorithm supported is SHA-1.The value is set to 0x01
 *   - Encryption Algorithm: The encryption algorithm in which this key is used. Currently support only one type: RSA. The value is set to 0x01.
 *   - HashValue: Which is calculated using SHA-1 over the following fields: RID & Index & Modulus & Exponent
 *   - Public Key Exponent: Actually, the real length of the exponent is either one byte or 3 bytes. It can have two values: 3 (Format is 0x00 00 00 03), or  65537 (Format is 0x00 01 00 01)
 *   - Modulus Length: LenL LenH Indicated the length of the next field.
 *   - Modulus: This is the modulus field of the public key. Its length is specified in the field above.
 * @param capkLen the length of capk data buffer
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
L2_setCAPK(IN BYTE * capk, IN int capkLen);

/**
 * Remove Certificate Authority Public Key
 *
 * Removes the CAPK as specified by the RID/Index
 *
 * @param capk 6 byte CAPK =  5 bytes RID + 1 byte INDEX
 * @param capkLen the length of capk data buffer
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
L2_removeCAPK(IN BYTE * capk, IN int capkLen);

/**
 * Remove All Certificate Authority Public Keys
 *
 * Removes all the CAPKs
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
L2_removeAllCAPK();

/**
 * Retrieve the Certificate Authority Public Key list
 *
 * Returns all the CAPK RID and Index installed on the terminal.
 *
 * @param keys [key1][key2]...[keyn], each key 6 bytes where
 *  key = 5 bytes RID + 1 byte index
 * @param keysLen the length of keys data buffer
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
L2_retrieveCAPKList(OUT BYTE * keys, IN_OUT int * keysLen);


/**
 * Set Certificate Revocation List Entry
 *
 * Sets the CRL into the list
 *
 * @param crl CRL format:
 *  [5 bytes RID][1 byte Index][3 bytes Serial Number]
 * @param crlLen the length of crl data buffer
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
L2_setCRLEntry(IN BYTE * crl, IN int crlLen);

/**
 * Remove a Certificate Revocation List Entry
 *
 * Removes the CRL from the list
 *
 * @param crl CRL format:
 *  [5 bytes RID][1 byte Index][3 bytes Serial Number]
 * @param crlLen the length of crl data buffer
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
L2_removeCRLEntry(IN BYTE * crl, IN int crlLen);

/**
 * Remove All CRL from the list
 *
 * Removes all the CRLs
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
L2_removeAllCRLEntries();

/**
 * Retrieve the Certificate revocation list
 *
 * Returns all the CRL’s on the list.
 *
 * @param keys [key1][key2]...[keyn], each key 9 bytes where
 *  key = 5 bytes RID + 1 byte index + 3 bytes serial number
 * @param keysLen the length of keys data buffer
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
L2_retrieveCRL(OUT BYTE * keys, IN_OUT int * keysLen);

/**
 * Polls device for EMV Kernel Version
 *
 * @param version Response returned of Kernel Version
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
L2_getEMVKernelVersion(OUT char * version);

/**
 * Get EMV Kernel check value info
 *
 * @param checkValue Response returned of Kernel check value info
 * @param checkValueLen the length of checkValue
 *
 * @return RETURN_CODE:	Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
L2_getEMVKernelCheckValue(OUT BYTE * checkValue, IN_OUT int * checkValueLen);

/**
 * Get EMV Kernel configuration check value info
 *
 * @param checkValue Response returned of Kernel configuration check value info
 * @param checkValueLen the length of checkValue
 * @return RETURN_CODE:	Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
L2_getEMVConfigurationCheckValue(OUT BYTE * checkValue, IN_OUT int * checkValueLen);


/**
 * Split a IDTech-TLV stream into 3 BER-TLV streams - unencrypted, encrypted, and masked.
 *
 * @param data - input data array containing the IDTech-TLV stream
 * @param startPos - starting index position of the IDTech-TLV stream
 * @param tlvLen - maximum length of the IDTech-TLV stream
 *
 * @param unencTLV - output unencrypted BER-TLV stream
 * @param unencLen - input and output length of the unencrypted BER-TLV stream
 * @param encTLV - output encrypted BER-TLV stream
 * @param encLen - input and output length of the encrypted BER-TLV stream
 * @param maskTLV - output masked BER-TLV stream
 * @param maskLen - input and output length of the masked BER-TLV stream
 * @return 0 if successful, 1 if unsuccessful
 */
int
tlv_processIDTTLV(IN BYTE * data, IN int startPos, IN int tlvLen, OUT BYTE * unencTLV, OUT int * unencLen,
  OUT BYTE * encTLV, OUT int * encLen, OUT BYTE * maskTLV, OUT int * maskLen);

/**
 * Sort a BER-TLV stream based on the tags from low to high.
 *
 * @param data - BER-TLV stream
 * @param tlvLen - length of the BER-TLV stream
 * @param recursive - 1= Sorted nested TLV streams within compound tags, 0= only sort top level
 *
 * @return 0 if sort was successful, 1 if unsuccessful
 */
int
tlv_sortTLV(IN_OUT BYTE * data, IN int tlvLen, IN int recursive);

/**
 * Get the value of a tag in the provided BER-TLV stream. Top level only, no retrieval within compound tags.
 * The TLV values in a compound tag should be submitted as a separate inquiry to Get Tag Value.
 *
 * @param data - BER-TLV stream
 * @param tlvLen - length of the BER-TLV stream
 * @param tag - the input tag for the value
 * @param tagLen � the length of the input tag
 *
 * @param value - output value
 * @param valueLen - length of output value
 * @param recursive � 1= Search nested TLV streams within compound tags, 0= only search top level
 *
 * @return 0 if successful, 1 if unsuccessful
 *
 */
int
tlv_getTagValue(IN BYTE * data, IN int tlvLen, IN BYTE * tag, IN int tagLen, OUT BYTE * value, OUT int * valueLen);

/**
 * Append a BER-TLV stream to another BER-TLV stream.
 *
 * @param data - main BER-TLV stream that will include the other stream
 * @param tlvLen - length of the main BER-TLV stream
 * @param tlvAdd - other BER-TLV stream that will be added to the main stream
 * @param tlvAddLen - length of the other BER-TLV stream
 * @param overWrite - if 1, tags in other stream will overwrite,
 *   if 0, tags in other stream will be ignored
 *
 * @return 0 if successful, 1 if unsuccessful
 */
int
tlv_addTLV(IN_OUT BYTE * data, IN_OUT int * tlvLen, IN BYTE * tlvAdd, IN int tlvAddLen, IN int overWrite);

/**
 * Remove a BER-TLV from a BER-TLV stream.
 *
 * @param data - BER-TLV stream
 * @param tlvLen - length of the BER-TLV stream
 * @param tag - the tag of the BER-TLV to remove
 * @param tagLen � the length of the input tag
 * @param recursive � 1= Search nested TLV streams within compound tags, 0= only search top level
 *
 * @return 0 if tag removed, else 1
 */
int
tlv_removeTLV(IN_OUT BYTE * data, IN_OUT int * tlvLen, IN BYTE * tag, IN int tagLen, IN int recursive);

/**
 * Count the number of BER-TLV in a BER-TLV stream.
 *
 * @param data - BER-TLV stream
 * @param tlvLen - length of the BER-TLV stream
 * @param numTags � Number of tags in the valid BER-TLV stream.
 *
 * @return 0 if successful, 1 if unsuccessful
 */
int
tlv_countTLV(IN BYTE * data, IN int tlvLen, OUT int * numTags);

/**
 *  Validates the data as a complete TLV stream (BER-TLV or IDTech Enhanced TLV).
 *
 * @param data - TLV stream
 * @param tlvLen - length of the BER-TLV stream
 * @param recursive � 1= Validate nested TLV streams within compound tags, 0= only validate top level
 *
 * @return 0 if valid TLV stream, else 1
 */
int
tlv_validateTLV(IN BYTE * data, IN int tlvLen, IN int recursive);


/**
 * Dummy function. Returns 0
 */
int
lcd_clearDisplay(IN BYTE control);


/**
 * Dummy function. Returns 0
 */
int
lcd_captureSignature(IN int timeout);

/**
 * Dummy function. Returns 0
 */
int
emv_getAutoAuthenticateTransaction();


/**
 * Dummy function. Returns 0
 */
int
device_enhancedPassthrough(IN BYTE * data, IN int dataLen);

/**
 * Dummy function. Returns 0
 */
int
lcd_customDisplayMode(IN int enable);

/**
 * Dummy function. Returns 0
 */
int
pin_promptCreditDebit(IN char * currencySymbol, IN int currencySymbolLen, IN char * displayAmount,
  IN int displayAmountLen, IN int timeout, OUT BYTE * retData, IN_OUT int * retDataLen);

/**
 * Dummy function. Returns 0
 */
int
pin_getEncryptedOnlinePIN(IN int keyType, IN int timeout);

/**
 * Dummy function. Returns 0
 */
void
emv_setAutoCompleteTransaction(IN int complete);


// do nothing, to remove demo purpose only code
int
emv_getAutoCompleteTransaction();


/**
 * Review the return code description.<br/>
 * @param returnCode  the response result.
 * @param description
 * @retval the string for description of response result
 *
 * - 0: "no error, beginning task";
 * - 1: "no response from reader";
 * - 2: "invalid response data";
 *
 * - 01: " Incorrect Header Tag";
 * - 02: " Unknown Command";
 * - 03: " Unknown Sub-Command";
 * - 04: " CRC Error in Frame";
 * - 05: " Incorrect Parameter";
 * - 06: " Parameter Not Supported";
 * - 07: " Mal-formatted Data";
 * - 08: " Timeout";
 * - 0A: " Failed / NACK";
 * - 0B: " Command not Allowed";
 * - 0C: " Sub-Command not Allowed";
 * - 0D: " Buffer Overflow (Data Length too large for reader buffer)";
 * - 0E: " User Interface Event";
 * - 10: " Need clear firmware(apply in boot loader only)";
 * - 11: " Communication type not supported, VT-1, burst, etc. Need encrypted firmware (apply in boot loader only)";
 * - 12: " Secure interface is not functional or is in an intermediate state.";
 * - 13: " Data field is not mod 8";
 * - 14: " Pad 0x80 not found where expected";
 * - 15: " Specified key type is invalid";
 * - 16: " Could not retrieve key from the SAM (InitSecureComm)";
 * - 17: " Hash code problem";
 * - 18: " Could not store the key into the SAM (InstallKey)";
 * - 19: " Frame is too large";
 * - 1A: " Unit powered up in authentication state but POS must resend the InitSecureComm command";
 * - 1B: " The EEPROM may not be initialized because SecCommInterface does not make sense";
 * - 1C: " Problem encoding APDU Module-Specific Status Codes ";
 * - 20: " Unsupported Index (ILM) SAM Transceiver error - problem communicating with the SAM (Key Mgr)";
 * - 21: " Unexpected Sequence Counter in multiple frames for single bitmap (ILM)Length error in data returned from the SAM (Key Mgr)
 * - 22: " Improper bit map (ILM)";
 * - 23: " Request Online Authorization";
 * - 24: " ViVOCard3 raw data read successful";
 * - 25: " Message index not available (ILM) ViVOcomm activate transaction card type (ViVOcomm)";
 * - 26: " Version Information Mismatch (ILM)";
 * - 27: " Not sending commands in correct index message index (ILM)";
 * - 28: " Time out or next expected message not received (ILM)";
 * - 29: " ILM languages not available for viewing (ILM)";
 * - 2A: " Other language not supported (ILM)";
 * - 41: " from 41 to 4F, Module-specific errors for Key Manager";
 *
 * - 50: " Auto-Switch OK";
 * - 51: " Auto-Switch failed";
 * - 70: " Antenna Error 80h Use another card";
 * - 81: " Insert or swipe card";
 * - 90: " Data encryption Key does not exist";
 * - 91: " Data encryption Key KSN exhausted";
 */
void
device_getIDGStatusCodeString(IN int returnCode, OUT char * despcrition);

void
comm_setCAPath(const char * path);

void
device_getResponseCodeString(IN int returnCode, OUT char * despcrition);

void
pin_registerCallBk(pPIN_callBack pPINf);

int
L2_setApplicationDataTLV(IN BYTE * tlv, IN int tlvLen);

int
emv_retrieveCAPKRIDIndexes(IN BYTE * RID, IN int RIDLen, OUT BYTE * tlv, IN_OUT int * tlvLen);

const char *
getCurrentDeviceName(char * currentDeviceName);

int
convertBytesToHexString(IN const unsigned char * hex_string, IN const int len, OUT char * dst, OUT int * dstLen);

int
convertHexStringToBytes(IN const char * str, unsigned char * bytes, int blen);

int
lcd_displayText(IN int posX, IN int posY, IN int displayWidth,
  IN int displayHeight, IN int fontDesignation, IN int fontID,
  IN int screenPosition, IN char * displayText, OUT BYTE * graphicsID);


# ifdef __cplusplus
}
# endif /* __cplusplus */

#endif /* _IDTECHSDK_H_ */
