#ifndef __LIBIDT_SREDKEY2_H___
# define __LIBIDT_SREDKEY2_H___


# define IN
# define OUT
# define IN_OUT
# include "IDTDef.h"

/**
 * Define the USB hot-plug callback function to monitor the info when
 * plug in/out the reader. <br/> It should be registered using the
 * registerHotplugCallBk, The first integer parameter is device type,
 * and the second integer parameter is either 0: Device Plugged Out or
 * 1: Device Plugged In
 */
typedef void (* pMessageHotplug)(int,
  int);

/**
 * Define the send command callback function to monitor the sending
 * command into the reader. <br/> It should be registered using the
 * registerLogCallBk
 */
typedef void (* pSendDataLog)(BYTE *,
  int);

/**
 * Define the read response callback function to monitor the reading
 * response from the reader. <br/> It should be registered using the
 * registerLogCallBk
 */
typedef void (* pReadDataLog)(BYTE *,
  int);

/**
 * Define the EMV callback function to get the transaction
 * message/data/result. <br/> It should be registered using the
 * emv_registerCallBk
 */
typedef void (* pEMV_callBack)(int,
  int,
  BYTE *,
  int,
  IDTTransactionData *,
  EMV_Callback *,
  int);

/**
 * Define the firmware update callback function to get the firmware
 * update status <br/> It should be registered using the
 * device_registerFWCallBk
 */
typedef void (* pFW_callBack)(int,
  int,
  int,
  int,
  int);

/**
 * Define the MSR callback function to get the MSR card data <br/>
 * It should be registered using the msr_registerCallBk, this callback function is for backward compatibility
 */
typedef void (* pMSR_callBack)(int, IDTMSRData);

/**
 * Define the MSR callback function to get pointer to the MSR card data <br/>
 * It should be registered using the msr_registerCallBk, this callback function is recommended instead of pMSR_callBack
 */
typedef void (* pMSR_callBackp)(int, IDTMSRData *);

/**
 * Define the PINPad callback function to get the input PIN Pad data <br/>
 * It should be registered using the pin_registerCallBk,
 */
typedef void (* pPIN_callBack)(int,
  IDTPINData *);

/**
 * Define the camera callback function to get the image data <br/>
 * It should be registered using the device_registerCameraCallBk,
 */
typedef void (* pCMR_callBack)(int,
  IDTCMRData *);

/**
 * Define the card status and front switch callback function to get card and front switch status <br/>
 * It should be registered using the device_registerCardStatusFrontSwitchCallBk,
 */
typedef void (* pCSFS_callBack)(BYTE status);

/**
 * Define the LCD callback function to get the input LCDItem <br/>
 * It should be registered using the lcd_registerCallBk,
 */
typedef void (* pLCD_callBack)(int,
  IDTLCDItem *);

/**
 * Define the comm callback function to get FTP file transfer status <br/>
 * It should be passed as a parameter in a FTP request,
 * Signature (int, int, int) = response code, current block, total blocks
 * RESPONSE CODES:
 *              100 = FILE DOWNLOAD STARTED
 *              101 = FILE BLOCK XX OF XX RECEIVED
 *              102 = FILE DOWNLOAD COMPLETED
 *              103 = FILE DOWNLOAD TERMINATED PREMATURELY
 *
 */
typedef void (* ftpComm_callBack)(int,
  int,
  int);

/**
 * Define the comm callback function to get the async url data <br/>
 * It should be registered using the comm_registerHTTPCallback
 */
typedef void (* httpComm_callBack)(BYTE *,
  int);

/**
 * Define the comm callback function to receive the V4 Protocol packets
 * received by the device from an external source (IP/USB/RS-232)
 * It should be registered using the comm_registerV4Callback,
 * Data callback will contain command, sub-command, and data from V4 packet
 */
typedef void (* v4Comm_callBack)(BYTE,
  BYTE,
  BYTE *,
  int);


# ifdef __cplusplus
extern "C" {
# endif

/**
 * To register the USB HID hot-plug callback function which implemented
 * in the application to monitor the hotplug message from the SDK.
 */
void
registerHotplugCallBk(pMessageHotplug pMsgHotplug);

/**
 * To register the log callback function which implemented in the
 * application to monitor sending/reading data between application and
 * reader.
 */
void
registerLogCallBk(pSendDataLog pFSend,
  pReadDataLog                 pFRead);

/**
 * To register the firmware update callback function to get the
 * firmware update processing response.  (Pass NULL to disable the callback.)
 */
void
device_registerFWCallBk(pFW_callBack pFWf);

/**
 * To register the camera callback function to get the image data.  (Pass NULL to disable the callback.)
 */
void
device_registerCameraCallBk(pCMR_callBack pCMRf);

/**
 * To register the card status and front switch callback function to get status.  (Pass NULL to disable the callback.)
 */
void
device_registerCardStatusFrontSwitchCallBk(pCSFS_callBack pCSFSf);

/**
 * To register the emv callback function to get the EMV processing response.  (Pass NULL to disable the callback.)
 */
void
emv_registerCallBk(pEMV_callBack pEMVf);

/**
 * To register the msr callback function to get the MSR card data.  (Pass NULL to disable the callback.)
 */
void
msr_registerCallBk(pMSR_callBack pMSRf);

/**
 * To register the msr callback function to get the MSR card data pointer.  (Pass NULL to disable the callback.)
 */
void
msr_registerCallBkp(pMSR_callBackp pMSRf);

/**
 * To register the ctls callback function to get the MSR card data.  (Pass NULL to disable the callback.)
 */
void
ctls_registerCallBk(pMSR_callBack pCTLSf);

/**
 * To register the ctls callback function to get the MSR card data pointer.  (Pass NULL to disable the callback.)
 */
void
ctls_registerCallBkp(pMSR_callBackp pCTLSf);

/**
 * To register the pin callback function to get the PINPad data.  (Pass NULL to disable the callback.)
 */
void
pin_registerCallBk(pPIN_callBack pPINf);

/**
 * To register the lcd callback function to get the LCDItem.  (Pass NULL to disable the callback.)
 */
void
lcd_registerCallBk(pLCD_callBack pLCDf);

/**
 * Register Comm HTTP Async Callback
 *
 * @param cBack - HTTP Comm callback
 */
void
comm_registerHTTPCallback(httpComm_callBack cBack);

/**
 * Register External V4 Protocol commands Callback
 *
 * @param cBack - V4 Protocol Comm callback
 */
void
comm_registerV4Callback(v4Comm_callBack cBack);

/**
 * To Get SDK version
 *@return return the SDK version string
 */
char *
SDK_Version();

/**
 * Set the path to use when searching for ID TECH's libraries.
 * If this is not set, the libraries will be searched for with the system's default procedures.
 *
 * @param absoluteLibraryPath The absolute path to ID TECH's libraries.
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
setAbsoluteLibraryPath(const char * absoluteLibraryPath);

/**
 * Set the path to the config xml file(s) if any <br/>
 *
 * @param path The path to the config xml files (such as "NEO2_Devices.xml"
 * which contains the information of NEO2 devices).
 *    Only need to specify the path to the folder which contains the config files.
 *    File names are not needed.
 *    The maximum length of path is 200 characters including the '\0' at the end.
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_setConfigPath(const char * path);

/**
 * Pass the content of the config xml file ("NEO2_Devices.xml") as a
 * string to the SDK instead of reading the config xml file by the SDK
 * It needs to be called before device_init(), otherwise the SDK will
 * try to read the config xml file.
 *
 * @param configs The content read from the config xml file ("NEO2_Devices.xml"
 *  which contains the information of NEO2 devices).
 * @param len The length of the string configs.  The maximum length is
 *  5000 bytes.
 */
int
device_setNEO2DevicesConfigs(IN const char * configs,
  IN int                                     len);

/**
 * Initial the device by USB<br/>
 * It will detect the device and trying connect. <br/>
 * The connect status can be checked by device_isConnected(). <br/>
 * Note: after the function returns success, the function device_setCurrentDevice() has to be called to set the device type.
 * @return RETURN_CODE: Values can be parsed with device_getResponseCodeString()
 */
int
device_init();

/**
 * Initial the device by RS232<br/>
 * It will try to connect to the device with provided deviceType, port_number, and brate. <br/>
 * @param deviceType Device to connect to
 * @param port_number Port number of the device
 *
 *   Port nr. |  Linux  | Windows
 *  -----------------------------
 *  | 0       | ttyS0   | COM1  |
 *  | 1       | ttyS1   | COM2  |
 *  | 2       | ttyS2   | COM3  |
 *  | 3       | ttyS3   | COM4  |
 *  | 4       | ttyS4   | COM5  |
 *  | 5       | ttyS5   | COM6  |
 *  | 6       | ttyS6   | COM7  |
 *  | 7       | ttyS7   | COM8  |
 *  | 8       | ttyS8   | COM9  |
 *  | 9       | ttyS9   | COM10 |
 *  | 10      | ttyS10  | COM11 |
 *  | 11      | ttyS11  | COM12 |
 *  | 12      | ttyS12  | COM13 |
 *  | 13      | ttyS13  | COM14 |
 *  | 14      | ttyS14  | COM15 |
 *  | 15      | ttyS15  | COM16 |
 *  | 16      | ttyUSB0 | n.a.  |
 *  | 17      | ttyUSB1 | n.a.  |
 *  | 18      | ttyUSB2 | n.a.  |
 *  | 19      | ttyUSB3 | n.a.  |
 *  | 20      | ttyUSB4 | n.a.  |
 *  | 21      | ttyUSB5 | n.a.  |
 *  | 22      | ttyAMA0 | n.a.  |
 *  | 23      | ttyAMA1 | n.a.  |
 *  | 24      | ttyACM0 | n.a.  |
 *  | 25      | ttyACM1 | n.a.  |
 *  | 26      | rfcomm0 | n.a.  |
 *  | 27      | rfcomm1 | n.a.  |
 *  | 28      | ircomm0 | n.a.  |
 *  | 29      | ircomm1 | n.a.  |
 *  | 30      | cuau0   | n.a.	|
 *  | 31      | cuau1   | n.a.  |
 *  | 32      | cuau2   | n.a.  |
 *  | 33      | cuau3   | n.a.  |
 *  | 34      | cuaU0   | n.a.  |
 *  | 35      | cuaU1   | n.a.  |
 *  | 36      | cuaU2   | n.a.  |
 *  | 37      | cuaU3   | n.a.  |
 *
 * @param brate Bitrate of the device
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString()
 */
int
rs232_device_init(int deviceType, int port_number, int brate);

/**
 * Sets the current device to talk to <br/>
 * The connect status can be checked by device_isConnected(). <br/>
 * @param deviceType Device to connect to
 * @code
 * enum DEVICE_TYPE {
 * IDT_DEVICE_UNKNOWN=0,
 * IDT_DEVICE_AUGUSTA_HID,
 * IDT_DEVICE_AUGUSTA_KB,
 * IDT_DEVICE_AUGUSTA_S_HID,
 * IDT_DEVICE_AUGUSTA_S_KB,
 * IDT_DEVICE_AUGUSTA_S_TTK_HID,
 * IDT_DEVICE_SPECTRUM_PRO,
 * IDT_DEVICE_MINISMART_II,
 * IDT_DEVICE_L80,
 * IDT_DEVICE_L100,
 * IDT_DEVICE_UNIPAY,
 * IDT_DEVICE_UNIPAY_I_V,
 * IDT_DEVICE_VP3300_AJ,
 * IDT_DEVICE_KIOSK_III,
 * IDT_DEVICE_KIOSK_III_S,
 * IDT_DEVICE_PIP_READER,
 * IDT_DEVICE_VENDI,
 * IDT_DEVICE_VP3300_USB,
 * IDT_DEVICE_UNIPAY_I_V_TTK,
 * IDT_DEVICE_VP3300_BT,
 * IDT_DEVICE_VP8800,
 * IDT_DEVICE_SREDKEY2_HID,
 * IDT_DEVICE_SREDKEY2_KB,
 * IDT_DEVICE_NEO2,
 * IDT_DEVICE_MINISMART_II_COM = IDT_DEVICE_NEO2+5,
 * IDT_DEVICE_SPECTRUM_PRO_COM,
 * IDT_DEVICE_KIOSK_III_COM,
 * IDT_DEVICE_KIOSK_III_S_COM,
 * IDT_DEVICE_VP3300_COM,
 * IDT_DEVICE_NEO2_COM,
 * IDT_DEVICE_MAX_DEVICES = IDT_DEVICE_NEO2_COM+5
 * };
 *      @endcode
 * @return RETURN_CODE:    1: success, 0: failed
 */
int
device_setCurrentDevice(int deviceType);

/**
 * Check if the device is attached to the USB port
 * The function device_init() must be called before this function.
 * @param deviceType, the device type of the USB device
 * @return 1 if the device is attached, or 0 if the device is not attached
 */
int
device_isAttached(int deviceType);

/**
 * Close the device <br/>
 *
 * @return RETURN_CODE:  0: success, 0x0A: failed
 */
int
device_close();

/**
 * Review the return code description.<br/>
 * @param returnCode  the response result.
 * @param description
 * @retval the string for description of response result
 *
 * - 0: "no error, beginning task";
 * - 1: "no response from reader";
 * - 2: "invalid response data";
 *
 * - 01: " Incorrect Header Tag";
 * - 02: " Unknown Command";
 * - 03: " Unknown Sub-Command";
 * - 04: " CRC Error in Frame";
 * - 05: " Incorrect Parameter";
 * - 06: " Parameter Not Supported";
 * - 07: " Mal-formatted Data";
 * - 08: " Timeout";
 * - 0A: " Failed / NACK";
 * - 0B: " Command not Allowed";
 * - 0C: " Sub-Command not Allowed";
 * - 0D: " Buffer Overflow (Data Length too large for reader buffer)";
 * - 0E: " User Interface Event";
 * - 10: " Need clear firmware(apply in boot loader only)";
 * - 11: " Communication type not supported, VT-1, burst, etc. Need encrypted firmware (apply in boot loader only)";
 * - 12: " Secure interface is not functional or is in an intermediate state.";
 * - 13: " Data field is not mod 8";
 * - 14: " Pad 0x80 not found where expected";
 * - 15: " Specified key type is invalid";
 * - 16: " Could not retrieve key from the SAM (InitSecureComm)";
 * - 17: " Hash code problem";
 * - 18: " Could not store the key into the SAM (InstallKey)";
 * - 19: " Frame is too large";
 * - 1A: " Unit powered up in authentication state but POS must resend the InitSecureComm command";
 * - 1B: " The EEPROM may not be initialized because SecCommInterface does not make sense";
 * - 1C: " Problem encoding APDU Module-Specific Status Codes ";
 * - 20: " Unsupported Index (ILM) SAM Transceiver error - problem communicating with the SAM (Key Mgr)";
 * - 21: " Unexpected Sequence Counter in multiple frames for single bitmap (ILM)Length error in data returned from the SAM (Key Mgr)
 * - 22: " Improper bit map (ILM)";
 * - 23: " Request Online Authorization";
 * - 24: " ViVOCard3 raw data read successful";
 * - 25: " Message index not available (ILM) ViVOcomm activate transaction card type (ViVOcomm)";
 * - 26: " Version Information Mismatch (ILM)";
 * - 27: " Not sending commands in correct index message index (ILM)";
 * - 28: " Time out or next expected message not received (ILM)";
 * - 29: " ILM languages not available for viewing (ILM)";
 * - 2A: " Other language not supported (ILM)";
 * - 41: " from 41 to 4F, Module-specific errors for Key Manager";
 *
 * - 50: " Auto-Switch OK";
 * - 51: " Auto-Switch failed";
 * - 70: " Antenna Error 80h Use another card";
 * - 81: " Insert or swipe card";
 * - 90: " Data encryption Key does not exist";
 * - 91: " Data encryption Key KSN exhausted";
 *
 */
void
device_getIDGStatusCodeString(IN int returnCode,
  OUT char *                         despcrition);

/**
 * Check the device conntected status
 * @return DEVICE_DISCONNECT=0, or DEVICE_CONNECTED = 1
 */
int
device_isConnected();

/**
 * DEPRECATED : please use device_getFirmwareVersion_Len(OUT char* firmwareVersion, IN_OUT int *firmwareVersionLen)
 *
 * Polls device for Firmware Version
 *
 * @param firmwareVersion Response returned of Firmware Version; needs to have at least 128 bytes of memory
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
device_getFirmwareVersion(OUT char * firmwareVersion);

/**
 * Polls device for Firmware Version
 *
 * @param firmwareVersion Response returned of Firmware Version
 * @param firmwareVersionLen Length of Firmware Version
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 *
 */
int
device_getFirmwareVersion_Len(OUT char * firmwareVersion, IN_OUT int * firmwareVersionLen);

/**
 * Ping Device
 *
 * Pings the reader.  If connected, returns success.  Otherwise, returns timeout.
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_pingDevice();

/**
 * Get current active device type
 * @return :  return the device type defined as DEVICE_TYPE in the IDTDef.h
 */
int
device_getCurrentDeviceType();

/**
 * Send a Command to NEO device
 *
 * Sends a command  to the NEO device .
 *
 * @param cmd  command to execute.
 * @param subCmd, sub command to execute.
 *
 * @param data buffer of NEO command data.
 * @param dataLen, the length of the buffer data.
 *
 * @param response Response data
 * @param respLen, the length of Response data
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_SendDataCommandNEO(IN int cmd,
  IN int                         subCmd,
  IN BYTE *                      data,
  IN int                         dataLen,
  OUT BYTE *                     response,
  IN_OUT int *                   respLen);

/**
 * Send a Command to NGA device
 *
 * Sends a command  to the device .
 *
 * @param cmd buffer of command to execute.
 * @param cmdLen, the length of the buffer cmd.
 *
 * @param data buffer of IDG command data.
 * @param dataLen, the length of the buffer data.
 *
 * @param response Response data
 * @param respLen, the length of Response data
 *
 * @return RETURN_CODE: Values can be parsed with device_getResponseCodeString()
 */
int
device_SendDataCommand(IN BYTE * cmd,
  IN int                         cmdLen,
  IN BYTE *                      data,
  IN int                         dataLen,
  OUT BYTE *                     response,
  IN_OUT int *                   respLen);

/**
 * Reboot Device
 * Executes a command to restart the device.
 * - Card data is cleared, resetting card status bits.
 * - Response data of the previous command is cleared.
 * - Resetting firmware.
 *
 * @return RETURN_CODE:  Values can be parsed with device_getIDGStatusCodeString()
 */
int
device_rebootDevice();

/**
 * Send a Command to ITP device
 *
 * Sends a command  to the device .
 *
 * @param cmd buffer of command to execute.
 * @param cmdLen, the length of the buffer cmd.
 *
 * @param data buffer of ITP command data.
 * @param dataLen, the length of the buffer data.
 *
 * @param response Response data
 * @param respLen, the length of Response data
 *
 * @return RETURN_CODE: Values can be parsed with device_getResponseCodeString()
 */
int
device_SendDataCommandITP(IN BYTE * cmd,
  IN int                            cmdLen,
  IN BYTE *                         data,
  IN int                            dataLen,
  OUT BYTE *                        response,
  IN_OUT int *                      respLen);

/**
 * Sets the transaction exponent to be used with device_startTransaction.  Default value is 2
 * @param exponent, The exponent to use when calling device_startTransaction
 */
void
device_setTransactionExponent(int exponent);

/**
 * Get Key Status
 *
 * Gets the status of loaded keys
 * @param status newFormat for Augusta and miniSmartII only
 *     1: new format of key status
 *     0: reserved format for support previous device
 * @param status
 *     For L80, L100, Augusta and miniSmartII:
 *        When the newFormat is 0, data format as follows.
 *        For Augusta and miniSmartII:
 *          byte 0:  PIN DUKPT Key, Does not support, always 0
 *          byte 1:  PIN Master Key, Does not support, always 0
 *          byte 2:  PIN Session Key, Does not support, always 0
 *          byte 3:  Account/MSR DUKPT Key, 1 Exists, 0 None, 0xFF STOP
 *          byte 4:  Account/ICC DUKPT Key, 1 Exists, 0 None, 0xFF STOP
 *          byte 5:  Admin DUKPT Key, 1 Exists, 0 None, 0xFF STOP
 *        For L80 and L100:
 *          byte 0:  PIN DUKPT Key
 *          byte 1:  PIN Master Key
 *          byte 2:  Standard PIN Session Key
 *          byte 3:  Desjardins PIN Session Key
 *          byte 4:  Account/MSR DUKPT Key, 1 Exists, 0 None, 0xFF STOP, Does not support, always 0
 *          byte 5:  Account/ICC DUKPT Key, 1 Exists, 0 None, 0xFF STOP, Does not support, always 0
 *          byte 6:  Admin DUKPT Key, 1 Exists, 0 None, 0xFF STOP
 *          byte 7:  Data DUKPT Key, 1 Exists, 0 None, 0xFF STOP
 *          byte 8:  MAC DUKPT Key, 1 Exists, 0 None, 0xFF STOP
 *
 *        when the newFormat is 1, data format as follows.
 *         [Block Length] [KeyStatusBlock1] [KeyStatusBlock2]...[KeyStatusBlockN]
 *        Where:
 *        [Block Length] is 2 bytes, format is Len_L Len_H, is KeyStatusBlock Number
 *        [KeyStatusBlockX> is 4 bytes, format is [Key Index and Key Name] [key slot] [key status]:
 *        [Key Index and Key Name] is 1 byte. Please refer to following table
 *            0x14    LCL-KEK to Encrypt Other Keys
 *            0x02    Data encryption Key to Encrypt ICC/MSR
 *            0x05    MAC DUKPT Key for Host-Device - MAC Verification
 *            0x05    MTK DUKPT Key for TTK Self-Test
 *            0x0C    RKI-KEK for Remote Key Injection
 *        [key slot] is 2 bytes. Range is 0 - 9999
 *        the MTK DUKPT Key slot is 16, the others are all 0
 *        [key status] is 1 byte.
 *            0 - Not Exist
 *            1 - Exist
 *        0xFF - (Stop. Only Valid for DUKPT Key)
 *    For NEO2 and SREDKey2:
 *		Each unit of three bytes represents one key's parameters (index and slot).
 *			Key Name Index (1 byte):
 *				0x14 - LCL-KEK
 *				0x01 - Pin encryption Key (NEO2 only)
 *				0x02 - Data encryption Key
 *				0x05 - MAC DUKPT Key
 *				0x0A - PCI Pairing Key (NEO2 only)
 *			Key Slot (2 bytes):
 *				Indicate different slots of a certain Key Name
 *					Example: slot =5 (0x00 0x05), slot=300 (0x01 0x2C)
 *					For BTPay380, slot is always 0
 *		For example, 0x14 0x00 0x00 0x02 0x00 0x00 0x0A 0x00 0x00 will represent
 *			[KeyNameIndex=0x14,KeySlot=0x0000], [KeyNameIndex=0x02,KeySlot=0x0000] and [KeyNameIndex=0x0A,KeySlot=0x0000]
 *
 * @param statusLen the length of status
 *
 * @return RETURN_CODE:    Values can be parsed with device_getResponseCodeString
 */
int
device_getKeyStatus(int * newFormat,
  BYTE *                  status,
  int *                   statusLen);

/**
 * Update Firmware
 * Updates the firmware of NEO 2 devices.
 * @param firmwareData Signed binary data of a firmware file provided by IDTech
 * @param firmwareDataLen Length of firmwareData
 * @param firmwareName Firmware name.
 *  - For example "VP5300_v1.00.023.0167.S_Test.fm"
 * @param encryptionType Encryption type
 * - 0 : Plaintext
 * - 1 : TDES ECB, PKCS#5 padding
 * - 2 : TDES CBC, PKCS#5, IV is all 0
 * @param keyBlob Encrypted firmware session key blob, TR-31 Rev B, wrapped by
 *  FW Key (Optional, none if firmware is plaintext)
 * @param keyBlobLen Length of keyBlob
 * @return RETURN_CODE:  Values can be parsed with errorCode.getErrorString()
 *
 * Firmware update status is returned in the callback with the following values:
 * sender = device type
 * state = DEVICE_FIRMWARE_UPDATE
 * current block
 * total blocks
 * ResultCode:
 * - RETURN_CODE_DO_SUCCESS = Firmware Update Completed Successfully
 * - RETURN_CODE_BLOCK_TRANSFER_SUCCESS = Current block transferred successfully
 * - Any other return code represents an error condition
 *
 */
int
device_updateFirmware(IN BYTE * firmwareData,
  IN int                        firmwareDataLen,
  IN char *                     firmwareName,
  IN int                        encryptionType,
  IN BYTE *                     keyBlob,
  IN int                        keyBlobLen);

/**
 * DEPRECATED : please use config_getModelNumber_Len(OUT char* sNumber, IN_OUT int *sNumberLen)
 *
 * Polls device for Model Number
 *
 * @param sNumber  Returns Model Number; needs to have at least 64 bytes of memory
 *
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString
 *
 */
int
config_getModelNumber(OUT char * sNumber);

/**
 * Polls device for Model Number
 *
 * @param sNumber  Returns Model Number
 * @param sNumber  length of Model Number
 *
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString
 *
 */
int
config_getModelNumber_Len(OUT char * sNumber, IN_OUT int * sNumberLen);

/**
 * DEPRECATED : please use config_getSerialNumber_Len(OUT char* sNumber, IN_OUT int *sNumberLen)
 *
 * Polls device for Serial Number
 *
 * @param sNumber  Returns Serial Number; needs to have at least 64 bytes of memory
 *
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString
 *
 */
int
config_getSerialNumber(OUT char * sNumber);

/**
 * Polls device for Serial Number
 *
 * @param sNumber  Returns Serial Number
 * @param sNumberLen  Length of Serial Number
 *
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString
 *
 */
int
config_getSerialNumber_Len(OUT char * sNumber, IN_OUT int * sNumberLen);

/**
 * Set System Language
 * Sets the language for the message displayed in the LCD screen
 *
 * @param language 2-byte ASCII code, can be "EN" or "JP"
 *
 * @return success or error code.  Values can be parsed with device_getIDGStatusCodeString
 * @see ErrorCode
 */
int
device_setSystemLanguage(char * language);

/**
 * Set Expiration Masking
 *
 * Sets the flag to mask the expiration date
 *
 * @param mask TRUE = mask expiration
 *
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString
 *
 */
int
msr_setExpirationMask(int mask);

/**
 * Get MSR expiration date mask.
 *
 * @param value
 *            5001 <Setting Value>.
 *              setting Value: '0' = masked, '1' = not-masked
 *
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString
 */
int
msr_getExpirationMask(BYTE * value);

/**
 * Set Clear PAN ID.
 *
 * @param val
 *            Set Clear PAN ID to value: Number of digits to show in clear.  Range 0-6.
 *
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString
 */
int
msr_setClearPANID(BYTE val);

/**
 * Get Clear PAN ID.
 *
 * Returns the number of digits that begin the PAN that will be in the clear
 *
 * @param value
 *            4901 <Setting Value>.
 *            setting Value: Number of digits in clear.  Values are char '0' - '6'
 *
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString
 */
int
msr_getClearPANID(BYTE * value);

/**
 * Set MSR Swipe Forced Encryption Option.
 *
 * @param tarck1
 *            Set track1 encryption to true or false.
 * @param tarck2
 *            Set track2 encryption to true or false.
 * @param tarck3
 *            Set track3 encryption to true or false.
 * @param tarck3card0
 *            Set track3 card0 encryption to true or false.
 *
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString
 */
int
msr_setSwipeForcedEncryptionOption(int track1, int track2, int track3, int track3card0);

/**
 * Get MSR Swipe Forced Encryption Option.
 *
 * @param option
 *            8401 <Setting Value>.
 *            Setting Value Byte using lower four bits as flags.  0 = Force Encryption Off, 1 = Force Encryption On
 *				   bit0 = Track 1
 *				   bit1 = Track 2
 *				   bit2 = Track 3
 *				   bit4 = Track 3 Card Option 0
 *
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString
 */
int
msr_getSwipeForcedEncryptionOption(BYTE * option);

/**
 * Set MSR Swipe Mask Option.
 *
 * Sets the swipe mask/clear data sending option
 *
 * @param tarck1
 *            Set track1 mask to true or false.
 * @param tarck2
 *            Set track2 mask to true or false.
 * @param tarck3
 *            Set track3 mask to true or false.
 *
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString
 */
int
msr_setSwipeMaskOption(int track1, int track2, int track3);

/**
 * Get MSR Swipe Mask Option.
 *
 * Gets the swipe mask/clear data sending option
 *
 * @param option
 *            8601 <Setting Value>.
 *            Setting Value Byte using lower three bits as flags.  0 = Mask Option Off, 1 = Mask Option On
 *				   bit0 = Track 1
 *				   bit1 = Track 2
 *				   bit2 = Track 3
 *				   Example: Response 0x03 = Track1/Track2 Masked Option ON, Track3 Masked Option Off
 *
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString
 */
int
msr_getSwipeMaskOption(BYTE * option);

/**
 * Get MSR Function Status.
 *
 * Gets the MSR function status
 *
 * @param enable: 1 = MSR enabled, 0 = MSR disabled
 * @param isBufferMode: 1 = buffer mode, 0 = auto mode
 * @param withNotification: 1 = with notification, 0 = without notification
 *
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString
 */
int
msr_getFunctionStatus(int * enable, int * isBufferMode, int * withNotification);

/**
 * Disable MSR
 * Disable MSR functions.
 * @return RETURN_CODE:  Values can be parsed with device_getResponseCodeString()
 */
int
msr_disable();

# ifdef __cplusplus
}
# endif


#endif // ifndef __LIBIDT_SREDKEY2_H___

/*! \file libIDT_SREDKey2.h
 * \brief SREDKey2 API.
 *
 * SREDKey2 Global API methods.
 */

/*! \def IN
 * INPUT parameter.
 */

/*! \def OUT
 * OUTPUT parameter.
 */

/*! \def IN_OUT
 * INPUT / OUTPUT PARAMETER.
 */

/*! \def _DATA_BUF_LEN
 * DATA BUFFER LENGTH
 */
